using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;

namespace Magify.Rx
{
    public static class ReactivePropertyExtensions
    {
        [NotNull]
        public static IObservable<T> SkipLatestValueOnSubscribe<T>([NotNull] this IReadOnlyReactiveProperty<T> source)
        {
            return source.HasValue ? source.Skip(1) : source;
        }

        public static UniTask<T>.Awaiter GetAwaiter<T>([NotNull] this IReadOnlyReactiveProperty<T> source)
        {
            return source.WaitUntilValueChangedAsync(CancellationToken.None).GetAwaiter();
        }

        [NotNull]
        public static UniTask<T> WaitUntilValueChangedAsync<T>([NotNull] this IReadOnlyReactiveProperty<T> source, CancellationToken cancellationToken)
        {
            var promise = new UniTaskCompletionSource<T>();
            var mediator = new SingleAssignmentDisposable();
            var shouldSkipValue = source.HasValue;

            mediator.Disposable = source.Subscribe(x =>
            {
                if (shouldSkipValue)
                {
                    shouldSkipValue = false;
                }
                else
                {
                    mediator.Dispose(); // cancel self (subscription)
                    promise.TrySetResult(x);
                }
            }, ex => promise.TrySetException(ex!), () => promise.TrySetCanceled());

            cancellationToken.Register(canceledHandler, false);
            return promise.Task!;

            void canceledHandler()
            {
                mediator?.Dispose();
                promise?.TrySetCanceled(cancellationToken);
            }
        }
    }
}