using System.Diagnostics;
using System.IO;
using System.Text.RegularExpressions;
using UnityEditor;
using UnityEngine;
using Debug = UnityEngine.Debug;

namespace Magify
{
    public static class EmbeddedCreativesEditorTools
    {
        private static readonly string[] _rootPlatformFolders =
        {
            "Android",
            "iOS"
        };

        private static bool _needPrepareCreatives;

        private static readonly string _uniqueTempPathInProject = FileUtil.GetUniqueTempPathInProject();

        public static void PrepareCreativesForBuild()
        {
            _needPrepareCreatives = HasMagifyFolder();
            if (!_needPrepareCreatives) return;

            CacheMagifyCreativesToTempFolder();
            CopyUniqueValidFileWithoutPrefixes();
            AssetDatabase.Refresh(ImportAssetOptions.ForceUpdate);
            AssetDatabase.SaveAssets();
        }

        public static void RestoreCreativesAfterBuild()
        {
            if (!_needPrepareCreatives) return;

            CopyMagifyCreativesToStreamingAssets();
            AssetDatabase.Refresh(ImportAssetOptions.ForceUpdate);
            AssetDatabase.SaveAssets();
        }

        private static void CacheMagifyCreativesToTempFolder()
        {
            var pathDefaultCreatives = GetPathToRootDefaultCreatives();
            var tempPath = GetTempPathForDefaultCreatives();
            CopyDirectory(pathDefaultCreatives, tempPath, true, true);
            Debug.Log($"Copy magify creatives to {tempPath}");

            var relativePath = GetAssetRelativePath(pathDefaultCreatives);
            AssetDatabase.DeleteAsset(relativePath);
            Directory.CreateDirectory(pathDefaultCreatives);
        }

        private static void CopyMagifyCreativesToStreamingAssets()
        {
            var pathDefaultCreatives = GetPathToRootDefaultCreatives();
            var tempPath = GetTempPathForDefaultCreatives();
            Directory.Delete(pathDefaultCreatives, true);
            CopyDirectory(tempPath, pathDefaultCreatives, true, true);
            Directory.Delete(tempPath, true);
            Debug.Log($"Copy magify creatives to {pathDefaultCreatives}");
        }

        private static string GetTempPathForDefaultCreatives()
        {
            return Path.Combine(_uniqueTempPathInProject, PackageInfo.RootDefaultCreatives);
        }

        private static bool HasMagifyFolder()
        {
            var pathDefaultCreatives = GetPathToRootDefaultCreatives();
            Debug.Log($"Check magify folder: {pathDefaultCreatives}");
            if (!Directory.Exists(pathDefaultCreatives))
            {
                Debug.Log("Magify folder didn't created");
                return false;
            }

            var pathToPlatformFolder = Path.Combine(pathDefaultCreatives, GetPlatformFolder());
            Debug.Log($"Check platform folder: {pathToPlatformFolder}");
            if (!Directory.Exists(pathToPlatformFolder))
            {
                Debug.Log("Platform folder didn't created");
                return false;
            }

            return true;
        }

        private static void CopyUniqueValidFileWithoutPrefixes()
        {
            var tempPlatformFolder = Path.Combine(GetTempPathForDefaultCreatives(), GetPlatformFolder());
            var rootPath = GetPathToRootDefaultCreatives();
            foreach (var file in Directory.GetFileSystemEntries(tempPlatformFolder, "*", SearchOption.AllDirectories))
            {
                var fileName = Path.GetFileName(file);
                var extension = Path.GetExtension(file);
                if (extension.Length == 0 || extension == ".meta" || fileName[0] == '.') continue;

                var newName = GetNewNameForCreative(fileName);
                if (newName == string.Empty) continue;

                var newPath = Path.Combine(rootPath, newName);
                if (File.Exists(newPath)) continue;

                File.Copy(file, newPath);
                Debug.Log($"Rename asset from {file} to {newPath}");
            }
        }

        private static string GetPlatformFolder()
        {
            var buildTarget = EditorUserBuildSettings.activeBuildTarget;
            switch (buildTarget)
            {
                case BuildTarget.Android:
                    return _rootPlatformFolders[0];
                case BuildTarget.iOS:
                    return _rootPlatformFolders[1];
                default:
                    Debug.LogError($"Platform {buildTarget} don't supported!");
                    return "";
            }
        }

        private static string GetAssetRelativePath(string absolutePath)
        {
            return Path.GetRelativePath(Path.GetDirectoryName(Application.dataPath), absolutePath);
        }

        private static string GetPathToRootDefaultCreatives()
        {
            return Path.Combine(Application.streamingAssetsPath, PackageInfo.RootDefaultCreatives);
        }

        private static string GetPathToAssetsJson()
        {
            return Path.Combine(Application.streamingAssetsPath, PackageInfo.DefaultCreativesConfig);
        }

        internal static string GetNewNameForCreative(string fileName)
        {
            // Example: non_consumable_6c5ca676e03fb9f93eaf8217b4bc9d5b_de729dd7-c300-4543-8a98-f10f7923e9d4_3ltVN5O.jpeg
            //          non_consumable_ee389b0e3cd730210f703fa78abf128f_20014996-ea9c-49ab-bc4d-e3d1f2504eac.jpeg
            var defaultNonConsumableMatch = Regex.Match(fileName, @"^[A-Za-z0-9]+_[A-Za-z0-9]+_[A-Za-z0-9]+_[\w-]+\.(jpe?g|png)$");
            if (defaultNonConsumableMatch.Success)
            {
                var newName = Regex.Replace(fileName, @"^[A-Za-z0-9]+_[A-Za-z0-9]+_[A-Za-z0-9]+_", "");
                // Return: de729dd7-c300-4543-8a98-f10f7923e9d4_3ltVN5O.jpeg
                //         20014996-ea9c-49ab-bc4d-e3d1f2504eac.jpeg

                return EmbeddedCreativesUtilities.RemovePostfixFromUrl(newName);
                // Return: de729dd7-c300-4543-8a98-f10f7923e9d4.png
                //         20014996-ea9c-49ab-bc4d-e3d1f2504eac.jpeg
            }

            // Example: bonus_6c5ca676e03fb9f93eaf8217b4bc9d5b_de729dd7-c300-4543-8a98-f10f7923e9d4_3ltVN5O.jpeg
            //          external_ee389b0e3cd730210f703fa78abf128f_20014996-ea9c-49ab-bc4d-e3d1f2504eac.jpeg
            var defaultNewsFeedMatch = Regex.Match(fileName, @"^[A-Za-z0-9]+_[A-Za-z0-9]+_[\w-]+\.(jpe?g|png)$");
            if (defaultNewsFeedMatch.Success)
            {
                var newNameForCreative = Regex.Replace(fileName, @"^[A-Za-z0-9]+_[A-Za-z0-9]+_", "");
                // Return: de729dd7-c300-4543-8a98-f10f7923e9d4_3ltVN5O.png
                //         20014996-ea9c-49ab-bc4d-e3d1f2504eac.jpeg

                return EmbeddedCreativesUtilities.RemovePostfixFromUrl(newNameForCreative);
                // Return: de729dd7-c300-4543-8a98-f10f7923e9d4.png
                //         20014996-ea9c-49ab-bc4d-e3d1f2504eac.jpeg
            }

            // Example: 665fe6d4ccb7116334c50c30bf41d576_66c31d39-10ba-4e17-ba75-1be59ed7d24c.png
            var defaultCreativeMatch = Regex.Match(fileName, @"^[A-Za-z0-9]+_[A-Za-z0-9-]+\.(jpe?g|png)$");
            if (defaultCreativeMatch.Success)
            {
                return Regex.Replace(fileName, @"^[A-Za-z0-9]+_", "");
                // Return: 66c31d39-10ba-4e17-ba75-1be59ed7d24c.png
            }

            // Example: 9b25669bed5f9b7cce91a862d8b59dc2_3f66df47-8c37-4f36-bf3b-c38fd3fed998_7I20Nk7.png
            defaultCreativeMatch = Regex.Match(fileName, @"^[A-Za-z0-9]+_[A-Za-z0-9-]+_[\w-]+\.(jpe?g|png)$");
            if (defaultCreativeMatch.Success)
            {
                var noPrefix = Regex.Replace(fileName, @"^[A-Za-z0-9]+_", "");
                // noPrefix = 3f66df47-8c37-4f36-bf3b-c38fd3fed998_7I20Nk7.png
                return EmbeddedCreativesUtilities.RemovePostfixFromUrl(noPrefix);
                // Return: 3f66df47-8c37-4f36-bf3b-c38fd3fed998.png
            }

            // Example: default_spot_main_lto_1.6016.png
            //          default_spot_main_lto_1.6016_8qFBbdE.png
            var defaultLtoSpotMatch = Regex.Match(fileName, @"^[\w]+\.[\w]+\.(jpe?g|png)$");
            if (defaultLtoSpotMatch.Success)
            {
                return Regex.Replace(fileName, @"\.[\w]+\.", ".");
                // Return: default_spot_main_lto_1.png
            }

            return string.Empty;
        }

        private static void CopyDirectory(string sourceDir, string destinationDir, bool recursive, bool overwrite = false)
        {
            // Get information about the source directory
            var dir = new DirectoryInfo(sourceDir);

            // Check if the source directory exists
            if (!dir.Exists)
            {
                throw new DirectoryNotFoundException($"Source directory not found: {dir.FullName}");
            }

            // Cache directories before we start copying
            var dirs = dir.GetDirectories();

            // Create the destination directory
            Directory.CreateDirectory(destinationDir);

            // Get the files in the source directory and copy to the destination directory
            foreach (var file in dir.GetFiles())
            {
                var targetFilePath = Path.Combine(destinationDir, file.Name);
                file.CopyTo(targetFilePath, overwrite);
            }

            // If recursive and copying subdirectories, recursively call this method
            if (recursive)
            {
                foreach (var subDir in dirs)
                {
                    var newDestinationDir = Path.Combine(destinationDir, subDir.Name);
                    CopyDirectory(subDir.FullName, newDestinationDir, true, overwrite);
                }
            }
        }
    }
}