#if UNITY_LOCALIZATION
using System.Collections.Generic;
using System.Linq;
using UnityEditor;
using UnityEngine;
using UnityEditor.Localization;
using UnityEngine.Localization;

namespace Magify
{
    [CanEditMultipleObjects]
    [CustomEditor(typeof(BundleLocalization))]
    public class BundleLocalizationEditor : Editor
    {
        private class Styles
        {
            public static readonly GUIContent AnyLocales = EditorGUIUtility.TrTextContent("Any Locales");
            public static readonly GUIContent IncludeLocales = EditorGUIUtility.TrTextContent("Include Locales");
            public static readonly GUIContent ExcludeLocales = EditorGUIUtility.TrTextContent("Exclude Locales");
            public static readonly GUIContent SelectAll = EditorGUIUtility.TrTextContent("Select all");
            public static readonly GUIContent DeselectAll = EditorGUIUtility.TrTextContent("Deselect all");
        }
        
        private readonly string _localesFoldoutKey = $"{nameof(BundledCreative)}.{nameof(_isLocalesFoldout)}";
        private readonly Dictionary<Locale, bool> _usedLocales = new Dictionary<Locale, bool>();

        private BundleLocalization _bundleLocalization;
        private bool _isLocalesFoldout;

        private void OnEnable()
        {
            _isLocalesFoldout = EditorPrefs.GetBool(_localesFoldoutKey, false);
            _bundleLocalization = (BundleLocalization)target;

            _bundleLocalization._includeLocales = RemoveMissingLocales(_bundleLocalization._includeLocales);
            _bundleLocalization._excludeLocales = RemoveMissingLocales(_bundleLocalization._excludeLocales);

            foreach (var locale in LocalizationEditorSettings.GetLocales())
            {
                _usedLocales.Add(locale, false);
            }

            foreach (var locale in GetActiveLocalesList())
            {
                _usedLocales[locale] = true;
            }
        }

        private void OnDisable()
        {
            EditorPrefs.SetBool(_localesFoldoutKey, _isLocalesFoldout);
        }

        public override void OnInspectorGUI()
        {
            DrawLocales();

            serializedObject.ApplyModifiedProperties();
        }

        private void DrawLocales()
        {
            var compatibleWithAnyLocales = GUILayout.Toggle(_bundleLocalization._compatibleWithAnyLocales, Styles.AnyLocales);
            if (_bundleLocalization._compatibleWithAnyLocales != compatibleWithAnyLocales)
            {
                _bundleLocalization._compatibleWithAnyLocales = compatibleWithAnyLocales;
                InverseUsedLocales();
                EditorUtility.SetDirty(_bundleLocalization);
            }
            
            EditorGUILayout.Space();
            EditorGUILayout.LabelField(_bundleLocalization._compatibleWithAnyLocales ? Styles.ExcludeLocales : Styles.IncludeLocales, EditorStyles.boldLabel);

            foreach (var locale in LocalizationEditorSettings.GetLocales())
            {
                var newValue = EditorGUILayout.ToggleLeft(locale.ToString(), _usedLocales[locale]);
                if (_usedLocales[locale] != newValue)
                {
                    _usedLocales[locale] = newValue;
                    ChangeLocaleInList(locale, newValue);
                    EditorUtility.SetDirty(_bundleLocalization);
                }
            }

            GUILayout.BeginHorizontal();
            if (GUILayout.Button(Styles.SelectAll, GUILayout.MaxWidth(80)))
            {
                ToggleAll(true);
                EditorUtility.SetDirty(_bundleLocalization);
            }

            if (GUILayout.Button(Styles.DeselectAll, GUILayout.MaxWidth(80)))
            {
                ToggleAll(false);
                EditorUtility.SetDirty(_bundleLocalization);
            }
            GUILayout.EndHorizontal();
        }

        private void InverseUsedLocales()
        {
            var activeLocalesList = GetActiveLocalesList();
            var passiveLocalesList = GetPassiveLocalesList();
            
            var availableLocales = LocalizationEditorSettings.GetLocales();
            activeLocalesList.Clear();
            activeLocalesList.AddRange(availableLocales.Except(passiveLocalesList));
            passiveLocalesList.Clear();
            
            foreach (var locale in availableLocales)
            {
                var isUsedLocale = !_usedLocales[locale];
                _usedLocales[locale] = isUsedLocale; 
            }
        }

        private void ToggleAll(bool value)
        {
            foreach (var locale in LocalizationEditorSettings.GetLocales())
            {
                _usedLocales[locale] = value;
                ChangeLocaleInList(locale, value);
            }
        }

        private void ChangeLocaleInList(Locale locale, bool isUsed)
        {
            var listUsedLocales = GetActiveLocalesList();
            if (isUsed)
            {
                if (!listUsedLocales.Contains(locale))
                {
                    listUsedLocales.Add(locale);
                }
            }
            else
            {
                listUsedLocales.Remove(locale);
            }
        }

        private List<Locale> GetActiveLocalesList()
        {
            return _bundleLocalization._compatibleWithAnyLocales ? _bundleLocalization._excludeLocales : _bundleLocalization._includeLocales;
        }
        
        private List<Locale> GetPassiveLocalesList()
        {
            return _bundleLocalization._compatibleWithAnyLocales ? _bundleLocalization._includeLocales : _bundleLocalization._excludeLocales;
        }

        private List<Locale> RemoveMissingLocales(List<Locale> locales)
        {
            return locales.Where(locale => locale).ToList();
        }
    }
}
#endif