using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Rx;
using Cysharp.Threading.Tasks;
using UnityEngine;

namespace Magify
{
    internal readonly struct StoredAppFeatureLoadingWorkerJob : IMagifyWorkerJob
    {
    }

    internal class StoredAppFeatureLoadingWorker : IMagifyWorker<StoredAppFeatureLoadingWorkerJob>, IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();
        [NotNull]
        private readonly FeaturesProvider _features;
        [NotNull]
        private readonly INetworkStatusProvider _network;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly Subject<Unit> _onStoredAppFeaturesLoadingRequested;
        [NotNull]
        private readonly IReactiveProperty<bool> _loadStoredAppFeaturesOnNetworkRestore;

        public StoredAppFeatureLoadingWorker(
            [NotNull] FeaturesProvider features,
            [NotNull] INetworkStatusProvider network,
            [NotNull] IReactiveProperty<bool> loadStoredAppFeaturesOnNetworkRestore,
            [NotNull] Subject<Unit> onStoredAppFeaturesLoadingRequested)
        {
            _features = features;
            _network = network;
            _loadStoredAppFeaturesOnNetworkRestore = loadStoredAppFeaturesOnNetworkRestore;
            _onStoredAppFeaturesLoadingRequested = onStoredAppFeaturesLoadingRequested;
        }

        public void DoJob([NotNull] StoredAppFeatureLoadingWorkerJob job)
        {
            if (_network.Reachability.Value is NetworkState.Reachable)
            {
                _loadStoredAppFeaturesOnNetworkRestore.Value = false;
                _features.LoadStoredAppFeatures(_disposables.GetOrCreateToken()).ContinueWith(releaseAll).Forget();
                _onStoredAppFeaturesLoadingRequested.OnNext(Unit.Default);
            }
            else
            {
                _loadStoredAppFeaturesOnNetworkRestore.Value = true;
            }
            return;

            void releaseAll([CanBeNull] IEnumerable<ContentHandle<StoredAppFeatureContent>> features)
            {
                if (features == null) return;
                foreach (var contentHandle in features)
                    contentHandle?.Dispose();
            }
        }

        public void Dispose()
        {
            _disposables.Release();
        }
    }
}