﻿using System;
using System.Globalization;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Networking;

namespace Magify
{
    internal class MicrosoftServerTimeProvider : IServerTimeProvider
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Network);

        public async UniTask<DateTime?> TryGetServerTime(CancellationToken cancellationToken)
        {
            const string serverUrl = "https://www.microsoft.com/";
            var request = UnityWebRequest.Head(serverUrl);
            if (request is null)
            {
                _logger.Log($"Failed to get web request from Microsoft Server from url: {serverUrl}");
                return null;
            }
            await request.SendWebRequest().WithCancellation(cancellationToken);

            if (request.result is not UnityWebRequest.Result.Success)
            {
                _logger.Log($"Failed to get server time from Microsoft with result: {request.result}, and error: {request.error}");
                return null;
            }

            try
            {
                var date = request.GetResponseHeader("date");
                _logger.Log($"Response time from Microsoft: {date}");
                if (DateTime.TryParseExact(date,
                        "ddd, dd MMM yyyy HH:mm:ss 'GMT'",
                        CultureInfo.InvariantCulture.DateTimeFormat,
                        DateTimeStyles.AssumeUniversal, out var dateTime))
                {
                    _logger.Log($"Parsed time from Microsoft: UTC: {dateTime.ToUniversalTime()}, Local: {dateTime.ToLocalTime()}");
                    return dateTime.ToUniversalTime();
                }
                _logger.Log("Failed to parsing server time from Microsoft");
                return null;
            }
            catch (Exception e)
            {
                _logger.LogError($"Exception during parsing server time from Microsoft: {e.Message}");
            }

            return null;
        }
    }
}