using System;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    public interface IServicePrefs
    {
        IReadOnlyReactiveProperty<int> PurchasesCounter { get; }
        IReadOnlyReactiveProperty<int> RewardedVideoCounter { get; }
        IReadOnlyReactiveProperty<int> InterstitialVideoCounter { get; }
        IReadOnlyReactiveProperty<bool> NativeRateUsShown { get; }
        IReadOnlyReactiveProperty<bool> ConsentGivenEventSent { get; }
        IReadOnlyReactiveProperty<string> BoughtSubscriptionProduct { get; }
        IReadOnlyReactiveProperty<SubscriptionStatus> SubscriptionStatus { get; }
        IReadOnlyReactiveProperty<DateTime> SubscriptionExpireDate { get; }
        IReadOnlyReactiveProperty<DateTime> SubscriptionBoughtDate { get; }
    }

    internal class ServicePrefs : IDisposable, IServicePrefs
    {
        private const string KeyPurchasesCounter = "purchases_counter";
        private const string KeyRewardedVideoCounter = "rewarded_video_counter";
        private const string KeyInterstitialVideoCounter = "interstitial_video_counter";
        private const string KeyNativeRateUsShown = "native_rate_us_shown";
        private const string KeyConsentGivenSent = "concent_given_sent";
        private const string KeyPauseTime = "pause_time";
        private const string KeySubscriptionProduct = "subscription_product";
        private const string KeySubscriptionStatus = "subscription_status";
        private const string KeySubscriptionExpireDate = "subscription_expire_date";
        private const string KeySubscriptionBoughtDate = "subscription_bought_date";

        private readonly CompositeDisposable _disposables = new CompositeDisposable();

        #region Prefs storage data

        [NotNull]
        public IReactiveProperty<int> PurchasesCounter { get; }
        IReadOnlyReactiveProperty<int> IServicePrefs.PurchasesCounter => PurchasesCounter;

        [NotNull]
        public IReactiveProperty<int> RewardedVideoCounter { get; }
        IReadOnlyReactiveProperty<int> IServicePrefs.RewardedVideoCounter => RewardedVideoCounter;

        [NotNull]
        public IReactiveProperty<int> InterstitialVideoCounter { get; }
        IReadOnlyReactiveProperty<int> IServicePrefs.InterstitialVideoCounter => InterstitialVideoCounter;

        [NotNull]
        public IReactiveProperty<bool> NativeRateUsShown { get; }
        IReadOnlyReactiveProperty<bool> IServicePrefs.NativeRateUsShown => NativeRateUsShown;

        [NotNull]
        public IReactiveProperty<bool> ConsentGivenEventSent { get; }
        IReadOnlyReactiveProperty<bool> IServicePrefs.ConsentGivenEventSent => ConsentGivenEventSent;

        [NotNull]
        public IReactiveProperty<SubscriptionStatus> SubscriptionStatus { get; }
        IReadOnlyReactiveProperty<SubscriptionStatus> IServicePrefs.SubscriptionStatus => SubscriptionStatus;

        [NotNull]
        public IReactiveProperty<string> BoughtSubscriptionProduct { get; }
        IReadOnlyReactiveProperty<string> IServicePrefs.BoughtSubscriptionProduct => BoughtSubscriptionProduct;

        [NotNull]
        public IReactiveProperty<DateTime> SubscriptionExpireDate { get; }
        IReadOnlyReactiveProperty<DateTime> IServicePrefs.SubscriptionExpireDate => SubscriptionExpireDate;

        [NotNull]
        public IReactiveProperty<DateTime> SubscriptionBoughtDate { get; }
        IReadOnlyReactiveProperty<DateTime> IServicePrefs.SubscriptionBoughtDate => SubscriptionBoughtDate;

        [NotNull]
        public IReactiveProperty<DateTime> PauseTime { get; }

        #endregion

        internal static ServicePrefs Create(string storagePath)
        {
            var storage = BinaryStorage
                .Construct(storagePath)
                .AddPrimitiveTypes()
                .ThrowOnLoadError()
                .SupportEnum<SubscriptionStatus>()
                .Build();

            var prefs = new ServicePrefs(storage);
            return prefs;
        }

        private ServicePrefs([NotNull] BinaryStorage storage)
        {
            storage.AddTo(_disposables);
            PurchasesCounter = storage.GetReactiveProperty<int>(KeyPurchasesCounter);
            RewardedVideoCounter = storage.GetReactiveProperty<int>(KeyRewardedVideoCounter);
            InterstitialVideoCounter = storage.GetReactiveProperty<int>(KeyInterstitialVideoCounter);
            NativeRateUsShown = storage.GetReactiveProperty<bool>(KeyNativeRateUsShown);
            ConsentGivenEventSent = storage.GetReactiveProperty<bool>(KeyConsentGivenSent);
            PauseTime = storage.GetReactiveProperty<DateTime>(KeyPauseTime);
            BoughtSubscriptionProduct = storage.GetReactiveProperty<string>(KeySubscriptionProduct);
            SubscriptionStatus = storage.GetReactiveProperty(KeySubscriptionStatus, Magify.SubscriptionStatus.Inactive);
            SubscriptionExpireDate = storage.GetReactiveProperty<DateTime>(KeySubscriptionExpireDate);
            SubscriptionBoughtDate = storage.GetReactiveProperty<DateTime>(KeySubscriptionBoughtDate);
        }

        void IDisposable.Dispose()
        {
            _disposables.Dispose();
        }
    }
}