using System;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public class NetworkStatus : INetworkStatusProvider, IDisposable
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private const float MinUpdateInterval = 1;
        private const float MaxUpdateInterval = 5;

        [NotNull]
        private PooledCompositeDisposable _disposables = new();
        private float _lastUpdate;

        [NotNull]
        public IReactiveProperty<NetworkState> Reachability { get; } = new ReactiveProperty<NetworkState>(NetworkState.NotReachable);

        internal NetworkStatus()
        {
            _lastUpdate = -MinUpdateInterval;
            CheckConnectionRoutine().Forget();
            Reachability
                .Subscribe(_ => { _logger.Log($"{nameof(NetworkState)} changed to {Reachability.Value}"); })
                .AddTo(_disposables);
        }

        private void UpdateNetwork()
        {
            if (Time.realtimeSinceStartup - _lastUpdate <= MinUpdateInterval) return;
            var reachable = Application.internetReachability != NetworkReachability.NotReachable;
            switch (Reachability.Value)
            {
                case NetworkState.Reachable when !reachable:
                    Reachability.Value = NetworkState.NotReachable;
                    break;
                case NetworkState.NotReachable when reachable:
                    Reachability.Value = NetworkState.Reachable;
                    break;
            }

            _lastUpdate = Time.realtimeSinceStartup;
        }

        private async UniTask CheckConnectionRoutine()
        {
            while (!_disposables.IsDisposed)
            {
                UpdateNetwork();
                while (_lastUpdate + MaxUpdateInterval > Time.realtimeSinceStartup)
                {
                    await UniTask.Yield();
                }
            }
        }

        public void Dispose()
        {
            _disposables.Release();
        }
    }
}