using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    public class AppStateService : IDisposable
    {
        [NotNull]
        private readonly ReactiveProperty<(SyncStateResult Result, DateTime Requested, DateTime Responded)?> _lastSaveState = new();
        [NotNull]
        private readonly ReactiveProperty<(SyncStateResult Result, DateTime Requested, DateTime Responded)?> _lastRestoreState = new();
        [NotNull]
        private readonly CompositeDisposable _disposable = new();

        /// <summary>
        /// Determines whether application state API is technically available
        /// </summary>
        public bool IsAvailable => !MagifyManager.CustomClientIdWasSet;

        /// <inheritdoc cref="MagifyManager.SyncStateEnabled"/>
        [NotNull]
        public IReactiveProperty<bool> SyncStateEnabled => MagifyManager.SyncStateEnabled;

        /// <inheritdoc cref="MagifyManager.Synchronization.AutoRestoreStateInfo"/>
        [NotNull]
        public IReadOnlyReactiveProperty<AutoRestoreStateInfo> AutoRestoreStateInfo => MagifyManager.Synchronization.AutoRestoreStateInfo;

        /// <remarks>
        /// Time in UTC
        /// </remarks>
        [NotNull]
        public IReadOnlyReactiveProperty<(SyncStateResult Result, DateTime Requested, DateTime Responded)?> LastSaveState => _lastSaveState;
        [NotNull]
        public IObservable<SyncStateResult> OnSaveStateFinishedState => _lastSaveState.Where(t => t != null).Select(t => t!.Value.Result)!;

        /// <remarks>
        /// Time in UTC. <br/>
        /// 'Requested' time might be equal to 'Responded' time if it was automatic restore. <br/>
        /// In the case of an automatic restore, we will only notify you of a successful restore.
        /// </remarks>
        [NotNull]
        public IReadOnlyReactiveProperty<(SyncStateResult Result, DateTime Requested, DateTime Responded)?> LastRestoreState => _lastRestoreState;

        /// <inheritdoc cref="MagifyManager.Synchronization.OnRestoreStateCompleted"/>
        /// <remarks>
        /// If you want to have more control over the result of the restore, use the return value <see cref="RestoreState"/>.
        /// </remarks>
        [NotNull]
        public IObservable<SyncStateResult> OnRestoreStateFinishedState => _lastRestoreState.Where(t => t != null).Select(t => t!.Value.Result)!;

        internal AppStateService()
        {
            if (!MagifyManager.CustomClientIdWasSet)
            {
                MagifyManager.Synchronization.OnRestoreStateCompleted
                    .Subscribe(result => _lastRestoreState.Value = (result, Requested: DateTime.UtcNow, Responded: DateTime.UtcNow))
                    .AddTo(_disposable);
            }
        }

        /// <inheritdoc cref="MagifyManager.Authorization.AuthorizeUser"/>
        public UniTask<bool> AuthorizeUser([NotNull] string provider, [NotNull] string token, CancellationToken cancellationToken, TimeSpan? timeout = null, DelayType delayType = DelayType.Realtime, PlayerLoopTiming loopTiming = PlayerLoopTiming.Update)
        {
            cancellationToken = LinkTimeout(timeout, cancellationToken, delayType, loopTiming);
            return MagifyManager.Authorization.AuthorizeUser(provider, token, cancellationToken);
        }

        /// <inheritdoc cref="MagifyManager.Authorization.ResetUserAuthorization"/>
        public void ResetUserAuthorization()
        {
            MagifyManager.Authorization.ResetUserAuthorization();
        }

        /// <inheritdoc cref="MagifyManager.Synchronization.SaveState"/>
        public async UniTask<SyncStateResult> SaveState(int? weight, CancellationToken cancellationToken, TimeSpan? timeout = null, DelayType delayType = DelayType.Realtime, PlayerLoopTiming loopTiming = PlayerLoopTiming.Update)
        {
            if (weight < 0)
            {
                throw new MagifyArgumentOutOfRangeException($"{nameof(SaveState)}(int? weight)", ", it mustn't be less than zero");
            }
            cancellationToken = LinkTimeout(timeout, cancellationToken, delayType, loopTiming);
            var startTime = DateTime.UtcNow;
            var result = await MagifyManager.Synchronization.SaveState(weight, cancellationToken);
            _lastSaveState.Value = (result, Requested: startTime, Responded: DateTime.UtcNow);
            return result;
        }

        /// <inheritdoc cref="MagifyManager.Synchronization.RestoreState"/>
        public async UniTask<SyncStateResult> RestoreState(int? weight, CancellationToken cancellationToken, TimeSpan? timeout = null, DelayType delayType = DelayType.Realtime, PlayerLoopTiming loopTiming = PlayerLoopTiming.Update)
        {
            if (weight < 0)
            {
                throw new MagifyArgumentOutOfRangeException($"{nameof(RestoreState)}(int? weight)", ", it mustn't be less than zero");
            }
            cancellationToken = LinkTimeout(timeout, cancellationToken, delayType, loopTiming);
            var startTime = DateTime.UtcNow;
            var result = await MagifyManager.Synchronization.RestoreState(weight, cancellationToken);
            _lastRestoreState.Value = (result, Requested: startTime, Responded: DateTime.UtcNow);
            return result;
        }

        private CancellationToken LinkTimeout(TimeSpan? timeout, CancellationToken cancellationToken, DelayType delayType, PlayerLoopTiming loopTiming)
        {
            if (timeout != null)
            {
                var cancellationTokenSource = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken, _disposable.GetOrCreateToken());
                try
                {
                    UniTask.Delay(timeout.Value, delayType, loopTiming, cancellationTokenSource.Token)
                        .ContinueWith(() => cancellationTokenSource.Cancel())
                        .Forget();
                }
                catch (OperationCanceledException)
                {
                    // Ignore.
                    // It means that caller cancelled the initial cancellation token.
                    // Result cancellation token will be cancelled by the chain.
                }
                cancellationToken = cancellationTokenSource.Token;
            }
            return cancellationToken;
        }

        void IDisposable.Dispose()
        {
            _disposable.Dispose();
            _lastSaveState.Dispose();
            _lastRestoreState.Dispose();
        }
    }
}