using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public class BaseSubscriptionService : IDisposable
    {
        [NotNull]
        private protected static readonly MagifyLogger Logger = MagifyLogger.Get(MagifyService.LogScope);

        [NotNull]
        private protected readonly ServicePrefs Prefs;
        [NotNull]
        private protected readonly ServiceTime Time;
        [NotNull]
        private protected readonly INetworkStatusProvider Network;
        [NotNull]
        private protected readonly Subject<Unit> SubscriptionRefreshedSubject = new Subject<Unit>();
        [NotNull]
        private protected readonly Subject<bool> SubscriptionTrialStatusReadSubject = new Subject<bool>();

        [CanBeNull]
        private protected IMinimalInAppStore InAppStore;
        [CanBeNull]
        private protected CancellationTokenSource UpdateCancellationToken;
        private protected bool FirstLoadPassed;
        private protected bool HotStartPassed;

// ReSharper disable InconsistentNaming
        [NotNull]
        public IObservable<Unit> OnSubscriptionRefreshed => SubscriptionRefreshedSubject;
        /// <summary>
        /// <b>True</b> if subscription is in trial period
        /// </summary>
        [NotNull]
        public IObservable<bool> OnSubscriptionTrialStatusRead => SubscriptionTrialStatusReadSubject;
// ReSharper restore InconsistentNaming

        internal BaseSubscriptionService([NotNull] ServicePrefs prefs, [NotNull] ServiceTime time, [NotNull] INetworkStatusProvider network)
        {
            Prefs = prefs;
            Time = time;
            Network = network;
        }

        internal virtual void SetPurchasingProvider([CanBeNull] IMinimalInAppStore inAppStore)
        {
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished -= PurchaseFinishedHandler;
            }

            InAppStore = inAppStore;
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished += PurchaseFinishedHandler;
            }
        }

        void IDisposable.Dispose()
        {
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished -= PurchaseFinishedHandler;
            }
        }

        internal void HandleAppFocus(bool hasFocus)
        {
            if (!hasFocus)
            {
                HotStartPassed = false;
                return;
            }
            if (InAppStore == null || UpdateCancellationToken != null || HotStartPassed || !FirstLoadPassed || Prefs.BoughtSubscriptionProduct.Value == null || !Network.IsNetworkReachable)
            {
                return;
            }
            HotStartPassed = true;
            RefreshSubscription(Prefs.BoughtSubscriptionProduct.Value);
        }

        public virtual void ProductFetchedHandler([NotNull] string productId)
        {
            Logger.Log("Subscription product fetched handler called.");
            if (UpdateCancellationToken != null || Prefs.BoughtSubscriptionProduct.Value != productId || HotStartPassed)
            {
                Logger.Log("First product fetch already handled, skipping.");
                return;
            }

            FirstLoadPassed = true;
            HotStartPassed = true;
            RefreshSubscription(productId);
        }

        internal virtual void RefreshSubscription([CanBeNull] string productId)
        {
            if (InAppStore == null)
            {
                Logger.LogError("InAppStore is not set. Subscription refresh is not possible.");
                return;
            }
            var info = InAppStore.LoadSubscriptionInfo(productId);
            Logger.Log("Subscription info loaded. Updating subscription status.");
            if (info == null)
            {
                markAsExpired();
            }
            else if (Time.IsNetworkTimeProtected.Value || Application.platform == RuntimePlatform.Android)
            {
                Logger.Log("Subscription info is available. Updating subscription status.");
                updateCurrentSubscription();
            }
            else
            {
                Logger.Log("Waiting for network time to be protected to update subscription status.");
                waitServerTimeAndUpdate().Forget();
            }

            async UniTaskVoid waitServerTimeAndUpdate() // ios specific logic
            {
                UpdateCancellationToken = new CancellationTokenSource();
                try
                {
                    await UniTask.WaitUntil(() => Time.IsNetworkTimeProtected.Value, cancellationToken: UpdateCancellationToken.Token);
                    UpdateCancellationToken.Cancel();
                    UpdateCancellationToken.Dispose();
                    UpdateCancellationToken = null;
                }
                catch (OperationCanceledException)
                {
                    Logger.Log("Subscription refresh canceled.");
                }
                Logger.Log("Server time is protected. Checking subscription status.");
                if (info.IsExpired(Time.UtcNow))
                {
                    markAsExpired();
                }
                else Logger.Log("Subscription is not expired.");
            }

            void updateCurrentSubscription()
            {
                Prefs.SubscriptionExpireDate.Value = info.ExpireDate;
                Prefs.SubscriptionStatus.Value = GetSubscriptionInfoStatus(info.IsFreeTrial, false);
                SubscriptionRefreshedSubject.OnNext(Unit.Default);
                SubscriptionTrialStatusReadSubject.OnNext(info.IsFreeTrial);
            }

            void markAsExpired()
            {
                Logger.Log("Marking subscription as expired.");
                Prefs.SubscriptionStatus.Value = Prefs.SubscriptionStatus.Value switch
                {
                    SubscriptionStatus.Paid or SubscriptionStatus.ExpiredPaid => SubscriptionStatus.ExpiredPaid,
                    SubscriptionStatus.Trial or SubscriptionStatus.ExpiredTrial => SubscriptionStatus.ExpiredTrial,
                    _ => SubscriptionStatus.Inactive,
                };
                SubscriptionRefreshedSubject.OnNext(Unit.Default);
            }
        }

        private void PurchaseFinishedHandler([NotNull] string productId, [NotNull] PurchaseInfo purchaseInfo)
        {
            if (purchaseInfo.SubscriptionInfo == null)
            {
                Logger.Log($"Skip trying to handle purchase of {productId} as subscription. Subscription info is null.");
                return;
            }
            if (purchaseInfo.Receipt == null)
            {
                Logger.LogError($"Saving new subscription failed. The requested product ({productId}) not purchased yet (receipt is empty).");
                return;
            }
            Logger.Log($"Saving new subscription: \nProduct id - {productId}, \nReceipt: {purchaseInfo.Receipt}");
            var info = purchaseInfo.SubscriptionInfo;
            if (purchaseInfo.TrustedPurchaseRecord != null)
            {
                MagifyManager.TrackExternalTrustedSubscriptionActivation(purchaseInfo.TrustedPurchaseRecord);
            }
            else if (purchaseInfo.SkipVerification)
            {
                MagifyManager.TrackExternalSubscriptionActivationWithoutVerification(
                    info.IsFreeTrial,
                    productId,
                    purchaseInfo.Price,
                    purchaseInfo.Currency,
                    purchaseInfo.TransactionId,
                    purchaseInfo.OriginalTransactionId,
                    purchaseInfo.Store);
            }
            else
            {
                MagifyManager.TrackExternalSubscriptionActivation(
                    info.IsFreeTrial,
                    productId,
                    purchaseInfo.Price,
                    purchaseInfo.Currency,
                    purchaseInfo.TransactionId,
                    purchaseInfo.PurchaseToken,
                    purchaseInfo.OriginalTransactionId,
                    purchaseInfo.Receipt.Payload,
                    purchaseInfo.Store);
            }
            Prefs.BoughtSubscriptionProduct.Value = productId;
            Prefs.SubscriptionExpireDate.Value = info.ExpireDate;
            Prefs.SubscriptionBoughtDate.Value = DateTime.UtcNow;
            Prefs.SubscriptionStatus.Value = GetSubscriptionInfoStatus(info.IsFreeTrial, info.IsExpired(Time.UtcNow));
            FirstLoadPassed = true;
            SubscriptionRefreshedSubject.OnNext(Unit.Default);
            UpdateCancellationToken?.Cancel();
            UpdateCancellationToken?.Dispose();
            UpdateCancellationToken = null;
        }

        protected static SubscriptionStatus GetSubscriptionInfoStatus(bool isTrial, bool isExpired)
        {
            return isTrial switch
            {
                true when isExpired => SubscriptionStatus.ExpiredTrial,
                true => SubscriptionStatus.Trial,
                false when isExpired => SubscriptionStatus.ExpiredPaid,
                false => SubscriptionStatus.Paid
            };
        }
    }
}