using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    public class PurchaserService : BasePurchaserService, IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly MagifySettings _settings;
        private readonly ServicePrefs _prefs;
        private readonly ProductsObtainer _obtainer;
        [NotNull]
        private readonly HashSet<string> _purchasingInProgress = new();

        public IReadOnlyList<InAppProduct> InAppProducts => MagifyManager.InAppProducts;
        public IReadOnlyList<SubscriptionProduct> SubscriptionProducts => MagifyManager.SubscriptionProducts;
        [CanBeNull]
        public new IInAppStore InAppStore { get; private set; }

        internal PurchaserService(MagifySettings settings, ServicePrefs prefs, ProductsObtainer obtainer)
        {
            _settings = settings;
            _prefs = prefs;
            _obtainer = obtainer;
        }

        void IDisposable.Dispose()
        {
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished -= ExternalPurchaseFinishedHandler;
                MagifyManager.OnConfigLoaded -= FetchDefaultProducts;
            }
        }
        /// <summary> <b>
        /// You should use SetPurchasingProvider(IInAppStore inAppStore) in full version of Magify SDK
        /// </b> </summary>
        public override void SetPurchasingProvider([CanBeNull] IMinimalInAppStore inAppStore)
        {
            _logger.LogWarning("You should use SetPurchasingProvider(IInAppStore inAppStore) in full version of Magify SDK");
            SetPurchasingProvider(inAppStore as IInAppStore);
        }

        public void SetPurchasingProvider([CanBeNull] IInAppStore inAppStore)
        {
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished -= ExternalPurchaseFinishedHandler;
                MagifyManager.OnConfigLoaded -= FetchDefaultProducts;
            }
            InAppStore = inAppStore;
            if (InAppStore != null)
            {
                MagifyManager.OnConfigLoaded += FetchDefaultProducts;
                InAppStore.OnPurchaseFinished += ExternalPurchaseFinishedHandler;
                FetchDefaultProducts();
            }
        }

        public async UniTask<ProductPrepareResult> PrepareProductAsync(ProductDef product, CancellationToken cancellationToken)
        {
            var timeout = (float)_settings.ProductFetchTimeout;
            if (timeout <= 0)
            {
                return ProductPrepareResult.Fail(ProductObtainFailReason.Timeout);
            }

            var promise = new UniTaskCompletionSource<ProductObtainFailReason?>();
            InAppStore.OnProductFetched += fetched;
            InAppStore.OnProductFetchFailed += failed;

            try
            {
                InAppStore.LoadProducts(new[] { product });
                var result = await promise.Task
                    .AttachExternalCancellation(cancellationToken)
                    .Timeout(TimeSpan.FromSeconds(timeout));

                return result == null ? ProductPrepareResult.Success() : ProductPrepareResult.Fail(result.Value);
            }
            catch (TimeoutException)
            {
                return ProductPrepareResult.Fail(ProductObtainFailReason.Timeout);
            }
            catch (OperationCanceledException)
            {
                throw;
            }
            catch (Exception e)
            {
                return ProductPrepareResult.Fail(ProductObtainFailReason.Unknown, e.GetType().Name);
            }
            finally
            {
                InAppStore.OnProductFetched -= fetched;
                InAppStore.OnProductFetchFailed -= failed;
            }

            void fetched(string productId)
            {
                promise.TrySetResult(null);
            }

            void failed(string productId, ProductObtainFailReason reason)
            {
                promise.TrySetResult(reason);
            }
        }

        public bool IsPurchasingInProgress([NotNull] string productId)
        {
            return _purchasingInProgress.Contains(productId);
        }

        private void SetPurchasingInProgress([NotNull] string productId, bool value)
        {
            if (value)
            {
                _purchasingInProgress.Add(productId);
            }
            else
            {
                _purchasingInProgress.Remove(productId);
            }
        }

        public UniTask<(ProductObtainFailReason? FailReason, PurchaseInfo Info)> PurchaseProductAsync(InAppProduct product, CancellationToken cancellationToken)
        {
            return PurchaseProductAsync((ProductDef)product, cancellationToken);
        }

        public UniTask<(ProductObtainFailReason? FailReason, PurchaseInfo Info)> PurchaseProductAsync(SubscriptionProduct product, CancellationToken cancellationToken)
        {
            return PurchaseProductAsync((ProductDef)product, cancellationToken);
        }

        private async UniTask<(ProductObtainFailReason? FailReason, PurchaseInfo Info)> PurchaseProductAsync([NotNull] ProductDef product, CancellationToken cancellationToken)
        {
            _logger.Log($"Obtaining {product.GetType().Name} ({product.Id}): to obtain an in-app product, we need to use IPurchasesProvider");
            var purchasePromise = new UniTaskCompletionSource<(ProductObtainFailReason? FailReason, PurchaseInfo Info)>();
            InAppStore.OnPurchaseFinished += purchased;
            InAppStore.OnPurchaseFailed += failed;
            try
            {
                SetPurchasingInProgress(product.Id, true);
                InAppStore.Purchase(product);
                var result = await purchasePromise.Task.AttachExternalCancellation(cancellationToken);
                return result;
            }
            catch (OperationCanceledException)
            {
                return (ProductObtainFailReason.UserCancelled, null);
            }
            catch (Exception e)
            {
                Debug.LogException(e);
                return (ProductObtainFailReason.Unknown, null);
            }
            finally
            {
                if (!cancellationToken.IsCancellationRequested)
                {
                    await UniTask.SwitchToMainThread(cancellationToken);
                }
                InAppStore.OnPurchaseFinished -= purchased;
                InAppStore.OnPurchaseFailed -= failed;
                SetPurchasingInProgress(product.Id, false);
            }

            void purchased(string productId, PurchaseInfo purchaseInfo)
            {
                _prefs.PurchasesCounter.Value++;
                purchasePromise.TrySetResult((null, purchaseInfo));
            }

            void failed(string productId, ProductObtainFailReason reason)
            {
                purchasePromise.TrySetResult((reason, null));
            }
        }

        private void FetchDefaultProducts()
        {
            if (InAppStore == null)
            {
                _logger.Log($"Cannot fetch products: {nameof(InAppStore)} is null");
                return;
            }
            if (TaskScheduler.IsMainThread)
            {
                fetch();
            }
            else
            {
                switchToMainThreadAndFetch().Forget();
            }
            return;

            async UniTaskVoid switchToMainThreadAndFetch()
            {
                await UniTask.SwitchToMainThread();
                fetch();
            }

            void fetch()
            {
                var products = new List<ProductDef>();
                products.AddRange(InAppProducts);
                products.AddRange(SubscriptionProducts);
                _logger.Log($"Fetching default products from store {InAppStore.GetType()}: {string.Join(", ", products)}");
                InAppStore.LoadProducts(products);
            }
        }

        private void ExternalPurchaseFinishedHandler(string productId, PurchaseInfo purchaseInfo)
        {
            if (productId == null)
            {
                _logger.LogError($"{nameof(ExternalPurchaseFinishedHandler)} cannot handle null {nameof(productId)}.");
                return;
            }
            if (purchaseInfo == null)
            {
                _logger.LogError($"{nameof(ExternalPurchaseFinishedHandler)} cannot handle null {nameof(purchaseInfo)}");
                return;
            }
            if (IsPurchasingInProgress(productId))
                return;
            var product = (ProductDef)InAppProducts.FirstOrDefault(c => c.Id == productId) ?? SubscriptionProducts.FirstOrDefault(c => c.Id == productId);
            switch (product)
            {
                case InAppProduct when purchaseInfo.IsRestored:
                    _logger.Log($"Handled restored in-app for {productId}.");
                    MagifyManager.TrackRestoredInAppFor(productId);
                    break;
                case InAppProduct when purchaseInfo.SkipVerification:
                    _logger.Log($"There is external purchase of {productId} detected (this one doesn't require verification)");
                    MagifyManager.TrackExternalInAppWithoutVerification(
                        product.Id,
                        purchaseInfo.Price,
                        purchaseInfo.Currency,
                        purchaseInfo.TransactionId,
                        purchaseInfo.OriginalTransactionId,
                        purchaseInfo.Store);
                    _obtainer.HandleExternalObtain(product, purchaseInfo);
                    break;
                case InAppProduct:
                    _logger.Log($"There is external purchase of {productId} detected");
                    MagifyManager.TrackExternalInAppFor(
                        product.Id,
                        purchaseInfo.Price,
                        purchaseInfo.Currency,
                        purchaseInfo.TransactionId,
                        purchaseInfo.PurchaseToken,
                        purchaseInfo.OriginalTransactionId,
                        purchaseInfo.Receipt?.Payload,
                        purchaseInfo.Store);
                    _obtainer.HandleExternalObtain(product, purchaseInfo);
                    break;
                case SubscriptionProduct when purchaseInfo.IsRestored:
                    _logger.Log($"Handled restored subscription for {productId}.");
                    MagifyManager.TrackRestoredSubscription(productId);
                    break;
                case SubscriptionProduct:
                    _logger.Log($"Handle subscription for {productId}. Tracking will be in {nameof(SubscriptionService)}");
                    _obtainer.HandleExternalObtain(product, purchaseInfo);
                    break;
                default:
                    _logger.LogError($"No information about product {productId} in Magify!");
                    break;
            }
        }
    }
}