using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;
using static Magify.MagifyManager;

namespace Magify
{
    public enum ClickHandleResult
    {
        Canceled,
        NothingToDo,
        ClosePopup
    }

    public enum PopupShowResult
    {
        NoInternet,
        NotFound,
        Failed,
        Showed,
    }

    public delegate UniTask<ClickHandleResult> ClickHandler<in TItem>(TItem item, bool closeAfterObtain);

    public delegate UniTask<ClickHandleResult> ClickHandler();

    public delegate UniTask<TPopup> PopupLoader<TPopup, TArgs>(IPopupsProvider provider, CancellationToken cancellationToken)
        where TPopup : class, IPopupBase<TArgs>
        where TArgs : PopupArgs;

    public class PopupsHandler
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly MagifySettings _settings;
        private readonly ILocalizer _localizer;
        private readonly IPopupsProvider _embeddedProvider;
        private readonly INetworkStatusProvider _network;

        [CanBeNull]
        private readonly IPopupsProvider _customProvider;

        internal PopupsHandler(
            [NotNull] MagifySettings settings,
            [NotNull] ILocalizer localizer,
            [NotNull] INetworkStatusProvider network,
            [NotNull] IPopupsProvider embeddedProvider,
            [CanBeNull] IPopupsProvider customProvider)
        {
            _settings = settings;
            _localizer = localizer;
            _embeddedProvider = embeddedProvider;
            _network = network;
            _customProvider = customProvider;
        }

        public async UniTask<T> DoUnderSpin<T>(float minTime, CancellationToken cancellationToken, Func<CancellationToken, UniTask<T>> action)
        {
            var (provider, popup) = await Load<IPopupBase<PopupArgs>, PopupArgs>((provider, token) => provider.LoadSpinner(token), cancellationToken);
            if (popup == null)
            {
                throw new Exception("Can't show spinner");
            }
            var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
            cancellationToken = cts.Token;
            try
            {
                popup.OnHideRequested += onHideRequested;

                var args = new PopupArgs();
                await popup.ShowAsync(args, cancellationToken);
                var time = Time.realtimeSinceStartup;
                var result = await action(cancellationToken);
                var balance = minTime - (Time.realtimeSinceStartup - time);
                if (balance > 0)
                {
                    await UniTask.Delay(TimeSpan.FromSeconds(balance), true, cancellationToken: cancellationToken);
                }
                await popup.HideAsync(cancellationToken);
                return result;
            }
            finally
            {
                popup.OnHideRequested -= onHideRequested;
                await UniTask.SwitchToMainThread();
                popup.Dispose();
                provider.UnloadPopup(popup);
            }

            void onHideRequested()
            {
                cts.Cancel();
            }
        }

        public UniTask<PopupShowResult> ShowPopup(
            string screenId,
            PopupArgs args,
            Action showCallback,
            CancellationToken cancellationToken,
            ClickHandler clickHandler)
        {
            return ShowPopup(args, showCallback, (provider, token) => provider.LoadPopup(screenId, token), clickHandler, cancellationToken);
        }

        public UniTask<PopupShowResult> ShowRateReview(
            PopupArgs args,
            Action showCallback,
            CancellationToken cancellationToken,
            ClickHandler clickHandler)
        {
            return ShowPopup(args, showCallback, (provider, token) => provider.LoadRateReviewPopup(token), clickHandler, cancellationToken);
        }

        public async UniTask<PopupShowResult> ShowInterPopupSplash(TimeSpan showDuration, CancellationToken cancellationToken)
        {
            var (provider, popup) = await Load<IPopupBase<PopupArgs>, PopupArgs>((provider, token) => provider.LoadInterPopupSplash(token), cancellationToken);
            if (popup == null)
            {
                throw new Exception("Can't show spinner");
            }
            try
            {
                var args = new PopupArgs();
                await popup.ShowAsync(args, cancellationToken);
                await UniTask.Delay(showDuration, true, cancellationToken: cancellationToken);
                await popup.HideAsync(cancellationToken);
                return PopupShowResult.Showed;
            }
            catch (OperationCanceledException)
            {
                throw;
            }
            catch (Exception e)
            {
                Debug.LogException(e);
                return PopupShowResult.Failed;
            }
            finally
            {
                await UniTask.SwitchToMainThread();
                popup.Dispose();
                provider.UnloadPopup(popup);
            }
        }

        public async UniTask<PopupShowResult> ShowImagePopup(
            ImageCreative creative,
            Action showCallback,
            CancellationToken cancellationToken,
            ClickHandler clickHandler)
        {
            var isPortrait = IsPortrait;

            // Take image url based on current orientation (landscape or portrait)
            var contentUrl = isPortrait ? creative.Portrait : creative.Landscape;

            // Check is image url valid or not
            if (string.IsNullOrEmpty(contentUrl))
            {
                return PopupShowResult.NotFound;
            }

            var needDownload = !Storage.ImageIsCached(contentUrl);
            if (needDownload && !_network.IsNetworkReachable && !Storage.HasDefaultCreative(contentUrl))
            {
                return PopupShowResult.NoInternet;
            }

            // Load creative from url
            var result = needDownload && _network.IsNetworkReachable
                ? await DoUnderSpin(1, cancellationToken, async ct => await Storage.LoadImage(contentUrl, _settings.ScreenLoadTimeout, ct))
                : await Storage.LoadImage(contentUrl, _settings.ScreenLoadTimeout, cancellationToken);

            if (result.Code != StorageResultCode.Success)
            {
                return PopupShowResult.Failed;
            }
            try
            {
                var args = new ImagePopup.Args(creative, result.Value)
                {
                    CloseButtonDelay = creative.ButtonCloseTimeout
                };
                return await ShowPopup(args, showCallback, (provider, token) => args.Creative.IsFullScreen ? provider.LoadFullscreenPopup(token) : provider.LoadImagePopup(token), clickHandler, cancellationToken);
            }
            finally
            {
                result.Dispose();
            }
        }

        public async UniTask<PopupShowResult> ShowCreativePopup(
            ICampaign campaign,
            CreativePopup.Args args,
            Action showCallback,
            CancellationToken cancellationToken,
            ClickHandler<ProductDef> clickHandler)
        {
            var (provider, popup) = await Load<IPopupWithProducts<CreativePopup.Args>, CreativePopup.Args>((provider, token) => provider.LoadCreativeWithProductsPopup(args.Creative.ScreenId, token), cancellationToken);
            if (popup == null)
            {
                return PopupShowResult.NotFound;
            }
            var hidePromise = new UniTaskCompletionSource();
            IDisposable campaignUpdatesSubscription = default;
            cancellationToken.Register(() => hidePromise.TrySetCanceled());
            try
            {
                args.Localizer = _localizer;
                popup.OnClicked += itemClicked;
                popup.OnHideRequested += hideRequested;
                campaignUpdatesSubscription = campaign.SubscribeOnUpdates(onCampaignUpdates);
                await popup.ShowAsync(args, cancellationToken);
                showCallback?.Invoke();
                await hidePromise.Task;
                await popup.HideAsync(cancellationToken);
                return PopupShowResult.Showed;
            }
            catch (OperationCanceledException)
            {
                throw;
            }
            catch (Exception e)
            {
                Debug.LogException(e);
                return PopupShowResult.Failed;
            }
            finally
            {
                popup.OnClicked -= itemClicked;
                await UniTask.SwitchToMainThread();
                campaignUpdatesSubscription?.Dispose();
                popup.Dispose();
                provider.UnloadPopup(popup);
            }

            void hideRequested()
            {
                hidePromise.TrySetResult();
            }

            void onCampaignUpdates(ICampaign updatedCampaign)
            {
                _logger.Log($"Campaign {updatedCampaign.Name} updated. Update products in creative");
                var campaignWithProducts = (ICampaignWithProducts)updatedCampaign;
                // ReSharper disable once AccessToDisposedClosure
                popup.UpdateProducts(campaignWithProducts.Products);
            }

            void itemClicked(ProductDef product)
            {
                itemClickedAsync(product).Forget();
            }

            async UniTaskVoid itemClickedAsync(ProductDef product)
            {
                // ReSharper disable once AccessToDisposedClosure
                var clickHandleResult = await clickHandler(product, popup.CloseAfterObtain);
                if (cancellationToken.IsCancellationRequested) return;
                if (clickHandleResult == ClickHandleResult.ClosePopup) hidePromise.TrySetResult();
            }
        }

        private async UniTask<PopupShowResult> ShowPopup<TPopup, TArgs>(TArgs args,
            Action showCallback,
            PopupLoader<TPopup, TArgs> popupLoader,
            ClickHandler clickHandler,
            CancellationToken cancellationToken)
            where TPopup : class, IPopup<TArgs>
            where TArgs : PopupArgs
        {
            var (provider, popup) = await Load(popupLoader, cancellationToken);
            if (popup == null)
            {
                return PopupShowResult.NotFound;
            }
            var hidePromise = new UniTaskCompletionSource();
            cancellationToken.Register(() => hidePromise.TrySetCanceled());
            try
            {
                args.Localizer = _localizer;
                popup.OnClicked += itemClicked;
                popup.OnHideRequested += hideRequested;
                await popup.ShowAsync(args, cancellationToken);
                showCallback?.Invoke();
                await hidePromise.Task;
                await popup.HideAsync(cancellationToken);
                return PopupShowResult.Showed;
            }
            catch (OperationCanceledException)
            {
                throw;
            }
            catch (Exception e)
            {
                Debug.LogException(e);
                return PopupShowResult.Failed;
            }
            finally
            {
                popup.OnClicked -= itemClicked;
                await UniTask.SwitchToMainThread();
                popup.Dispose();
                provider.UnloadPopup(popup);
            }

            void hideRequested()
            {
                hidePromise.TrySetResult();
            }

            void itemClicked()
            {
                itemClickedAsync().Forget();
            }

            async UniTaskVoid itemClickedAsync()
            {
                var res = await clickHandler();
                if (cancellationToken.IsCancellationRequested) return;
                switch (res)
                {
                    case ClickHandleResult.ClosePopup:
                        hidePromise.TrySetResult();
                        break;
                }
            }
        }

        private async UniTask<(IPopupsProvider Provider, TPopup)> Load<TPopup, TArgs>(PopupLoader<TPopup, TArgs> getter, CancellationToken cancellationToken)
            where TPopup : class, IPopupBase<TArgs>
            where TArgs : PopupArgs
        {
            TPopup panel;
            if (_customProvider != null)
            {
                panel = await getter(_customProvider, cancellationToken);
                if (cancellationToken.IsCancellationRequested && panel != null)
                {
                    _customProvider.UnloadPopup(panel);
                }
                cancellationToken.ThrowIfCancellationRequested();
                if (panel != null) return (_customProvider, panel);
            }
            panel = await getter(_embeddedProvider, cancellationToken);
            if (cancellationToken.IsCancellationRequested && panel != null)
            {
                _embeddedProvider.UnloadPopup(panel);
            }
            cancellationToken.ThrowIfCancellationRequested();
            return (_embeddedProvider, panel);
        }
    }
}