#if !MAGIFY_ADVANCED
using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public partial class MagifyService
    {
        [NotNull]
        private readonly ServicePrefs _prefs;
        [NotNull]
        private readonly MagifySettings _settings;
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LogScope);
        public const string LogScope = "Service";

        [NotNull]
        private readonly IReactiveProperty<int> _sessionCounter;
        [NotNull]
        private readonly MagifyWorkersManager _workersManager;
        [NotNull]
        private readonly CompositeDisposable _disposables = new();
        [NotNull]
        private readonly CancellationTokenSource _terminateCancellationToken = new();
        [NotNull]
        private readonly Subject<Unit> _onStoredAppFeaturesLoadingRequested = new();
        [NotNull]
        private readonly IReactiveProperty<bool> _loadStoredAppFeaturesOnNetworkRestore = new ReactiveProperty<bool>();
        private int _configSyncsHandled = 0;

        public IObservable<Unit> OnConfigSynced { get; }
        [NotNull]
        internal IObservable<Unit> OnStoredAppFeaturesLoadingRequested => _onStoredAppFeaturesLoadingRequested;

        [NotNull]
        public IServicePrefs Prefs => _prefs;

        [NotNull]
        public INetworkStatusProvider Network { get; }

        [NotNull]
        public ServiceTime Time { get; }

        [NotNull]
        public BaseAnalyticsService Analytics { get; }

        [NotNull]
        public FeaturesProvider Features { get; }

        [NotNull]
        public BaseAdvertiserService Advertiser { get; }

        [NotNull]
        public BasePurchaserService Purchaser { get; }

        [NotNull]
        public BaseSubscriptionService Subscription { get; private set; }

        [NotNull]
        internal AdPreloader AdPreloader { get; }

        [NotNull]
        public string ClientId => MagifyManager.ClientId;

        /// <summary>
        /// Stores the current session number
        /// </summary>
        [NotNull]
        public IReadOnlyReactiveProperty<int> Session => _sessionCounter;

        /// <summary>
        /// Represents the current interval between sessions, that will be used to determine
        /// the new session start when player returns to the game after a long pause (when app is in background).
        /// </summary>
        public IReactiveProperty<TimeSpan> SessionsInterval => MagifyManager.SessionsInterval;

        /// <summary>
        /// Represents the current environment in which your application is running.
        /// It can have one of two values:
        /// <see cref="Environment.Production"/> for the release version of your app or
        /// <see cref="Environment.Staging"/> during developing and testing.
        /// </summary>
        public IReactiveProperty<Environment> Environment { get; }

        [NotNull]
        public IReactiveProperty<ConfigScope> RemoteConfigScopes => MagifyManager.RemoteConfigScopes;

        /// <summary>
        /// Switches the preloading of stored app features. SDK might load them in background on different triggers:
        /// <ul>
        ///     <li>start of new session</li>
        ///     <li>changing of network reachability</li>
        ///     <li><i>some other triggers...</i></li>
        /// </ul>
        /// </summary>
        public bool StoredAppFeaturesPreloadingEnabled { get; set; }

        /// <summary>
        /// This property determines whether logging is turned on or off in the SKD
        /// </summary>
        [NotNull]
        public IReactiveProperty<bool> IsLoggingEnabled { get; }

        /// <inheritdoc cref="MagifyManager.ExternalPurchaseVerificationHandler"/>
        [NotNull]
        public IReactiveProperty<IPurchaseVerificationHandler> ExternalPurchaseVerificationHandler => Purchaser.ExternalPurchaseVerificationHandler;

        /// <inheritdoc cref="MagifyManager.VerificationRetryInterval"/>
        [NotNull]
        public IReactiveProperty<float> VerificationRetryInterval => Purchaser.VerificationRetryInterval;

        internal MagifyService([NotNull] MagifySettings settings, [CanBeNull] MagifyServiceArgs args, [NotNull] ServicePrefs prefs)
        {
            _settings = settings;
            _prefs = prefs.AddTo(_disposables)!;

            // Create internal services
            Time = new ServiceTime().AddTo(_disposables)!;
            Network = new NetworkStatus().AddTo(_disposables)!;
            AdPreloader = new AdPreloader().AddTo(_disposables)!;
            Analytics = new BaseAnalyticsService().AddTo(_disposables)!;
            Features = new FeaturesProvider(_settings).AddTo(_disposables);
            Advertiser = new BaseAdvertiserService(_settings, prefs, null, AdPreloader, Network).AddTo(_disposables)!;
            Purchaser = new BasePurchaserService().AddTo(_disposables)!;
            Subscription = new BaseSubscriptionService(_prefs, Time, Network).AddTo(_disposables)!;

            // Setup internal events
            OnConfigSynced = Observable.FromEvent(t => MagifyManager.OnConfigLoaded += t, t => MagifyManager.OnConfigLoaded -= t);

            _sessionCounter = new ReactiveProperty<int>(MagifyManager.SessionNumber);
            Environment = new ReactiveProperty<Environment>(MagifyManager.Environment);
            Observable
                .FromEvent<int>(t => MagifyManager.OnSessionChanged += t, t => MagifyManager.OnSessionChanged -= t)
                .Subscribe(session => _sessionCounter.Value = session);

            _prefs.SubscriptionStatus
                .SkipLatestValueOnSubscribe()
                .Where(c => MagifyManager.SubscriptionStatus != c)
                .Subscribe(SubscriptionStatusChangedHandler)
                .AddTo(_disposables);
            Network.Reachability
                .SkipLatestValueOnSubscribe()
                .Where(c => c is NetworkState.Reachable)
                .Subscribe(NetworkReachabilityChangedHandler)
                .AddTo(_disposables);
            OnConfigSynced
                .Subscribe(ConfigSyncedHandler)
                .AddTo(_disposables);
            Environment
                .SkipLatestValueOnSubscribe()
                .Where(c => MagifyManager.Environment != c)
                .Subscribe(EnvironmentChangedHandler)
                .AddTo(_disposables);
            Session
                .Where(_ => StoredAppFeaturesPreloadingEnabled)
                .Subscribe(_ => _workersManager.DoJob<StoredAppFeatureLoadingWorkerJob>())
                .AddTo(_disposables);
            IsLoggingEnabled = new ReactiveProperty<bool>(MagifyManager.Logging.IsLoggingEnabled);
            IsLoggingEnabled
                .SkipLatestValueOnSubscribe()
                .Subscribe(c => MagifyManager.Logging.IsLoggingEnabled = c)
                .AddTo(_disposables);
            ApplicationHelper.EveryApplicationFocus()
                .Where(hasFocus => hasFocus)
                .ObserveOnMainThread(_disposables.GetOrCreateToken())
                .Subscribe(_ => _workersManager.DoJob<HandlePauseTimeOnFocusGainedWorkerJob>())
                .AddTo(_disposables);
            ApplicationHelper.EveryApplicationFocus()
                .ObserveOnMainThread(_disposables.GetOrCreateToken())
                .Subscribe(hasFocus => Subscription!.HandleAppFocus(hasFocus))
                .AddTo(_disposables);
            ApplicationHelper.EveryApplicationFocus()
                .Where(hasFocus => !hasFocus)
                .Subscribe(_ => _prefs.PauseTime.Value = DateTime.UtcNow)
                .AddTo(_disposables);

            var workers = new IMagifyWorker[]
            {
                new HandlePauseTimeOnFocusGainedWorker(_prefs, _settings, Time),
                new StoredAppFeatureLoadingWorker(Features, Network, _loadStoredAppFeaturesOnNetworkRestore, _onStoredAppFeaturesLoadingRequested),
            };
            _workersManager = new MagifyWorkersManager(workers).AddTo(_disposables);
        }

        /// <inheritdoc cref="BaseAdvertiserService.SetAdsMediator"/>
        [NotNull]
        public MagifyService SetAdsMediator([CanBeNull] IMinimalAdsMediator mediator)
        {
            _logger.Log($"Use ads mediator of type {mediator?.GetType().Name}");
            Advertiser.SetAdsMediator(mediator);
            return this;
        }

        /// <summary>
        /// Set the purchasing provider for the service. This method should be called right after the service initialization.
        /// </summary>
        /// <param name="inAppStore">Implementation that will be used to track in-app products and subscriptions purchases</param>
        [NotNull]
        public MagifyService SetPurchasingProvider([CanBeNull] IMinimalInAppStore inAppStore)
        {
            _logger.Log($"Use purchasing provider of type {inAppStore?.GetType().Name}");
            Purchaser.SetPurchasingProvider(inAppStore);
            Subscription.SetPurchasingProvider(inAppStore);
            return this;
        }

#if MAX_ADS_MEDIATOR
        /// <summary>
        /// Use embedded Max Mediator implementation. <br/>
        /// See: <see cref="MaxMediator"/>  (it's already implements <see cref="IAdsMediator"/>, so you can use it to show ads)
        /// </summary>
        [NotNull]
        public MagifyService UseEmbeddedMaxMediator()
        {
            _logger.Log($"Use embedded {nameof(MaxMediator)}");
            return SetAdsMediator(new MaxMediator(_settings));
        }

        /// <summary>
        /// It will configure the embedded <see cref="MaxMediator"/> implementation before tracking and showing of ads.
        /// </summary>
        [NotNull]
        public MagifyService InitializeEmbeddedMaxMediator()
        {
            _logger.Log($"Initialize embedded {nameof(MaxMediator)}");
            ((MaxMediator)Advertiser.Mediator!).Initialize();
            return this;
        }
#endif

#if LEVELPLAY_MEDIATOR
        /// <summary>
        /// Use embedded Level Play Mediator implementation. <br/>
        /// See: <see cref="LevelPlayMediator"/> (it's already implements <see cref="IAdsMediator"/>, so you can use it to show ads)
        /// </summary>
        [NotNull]
        public MagifyService UseEmbeddedLevelPlayMediator()
        {
            _logger.Log($"Use embedded {nameof(LevelPlayMediator)}");
            return SetAdsMediator(new LevelPlayMediator(_settings));
        }

        /// <summary>
        /// It will configure the embedded <see cref="LevelPlayMediator"/> implementation before tracking and showing of ads.
        /// </summary>
        [NotNull]
        public MagifyService InitializeEmbeddedLevelPlayMediator(bool testSuite = false)
        {
            _logger.Log($"Initialize embedded {nameof(LevelPlayMediator)}");
            ((LevelPlayMediator)Advertiser.Mediator!).Initialize(testSuite);
            return this;
        }
#endif

        /// <summary>
        /// You should call this method after showing AppTrackingTransparency dialog to set the user's choice (iOS only).
        /// </summary>
        public void SetAttStatus(bool authorized)
        {
            MagifyManager.SetAttStatus(authorized);
        }

        public void SetMediaSource([CanBeNull] string networkName = null, [CanBeNull] string campaignName = null, [CanBeNull] string adGroup = null)
        {
            MagifyManager.SetMediaSource(networkName, campaignName, adGroup);
        }

        public void Sync()
        {
            MagifyManager.Sync();
        }

        public void Reset(bool clearNativeStorage = true, bool clearCloudStorage = true)
        {
            MagifyManager.Reset(clearNativeStorage, clearCloudStorage);
        }

        private void ShutDown()
        {
            _terminateCancellationToken.Cancel();
            _terminateCancellationToken.Dispose();
            _disposables.Clear();
            MagifyManager.ShutDown();
        }

        private void SubscriptionStatusChangedHandler(SubscriptionStatus subscription)
        {
            _logger.Log($"Change SubscriptionStatus to {subscription}");
            MagifyManager.SubscriptionStatus = subscription;
            MagifyManager.Sync();
        }

        private void EnvironmentChangedHandler(Environment environment)
        {
            _logger.Log($"Change Environment to {environment}");
            MagifyManager.Environment = environment;
            MagifyManager.Sync();
        }

        private void NetworkReachabilityChangedHandler(NetworkState state)
        {
            _logger.Log($"Network reachability changed to {state} - call {nameof(MagifyManager)}.{nameof(MagifyManager.Sync)}");
            MagifyManager.Sync();
            if (_loadStoredAppFeaturesOnNetworkRestore.Value)
            {
                _workersManager.DoJob<StoredAppFeatureLoadingWorkerJob>();
            }
        }

        private void ConfigSyncedHandler<T>(T _)
        {
            _configSyncsHandled++;
            _logger.Log($"{nameof(ConfigSyncedHandler)} called ({_configSyncsHandled} call)");
            if (StoredAppFeaturesPreloadingEnabled && _configSyncsHandled == 1)
            {
                _workersManager.DoJob<StoredAppFeatureLoadingWorkerJob>();
            }
        }

        #region Not implemented in Base

        internal LimitedTimeOfferProvider Offers => throw new NotImplForMagifyBaseException(nameof(Offers));
        internal MagifyPresenter Presenter => throw new NotImplForMagifyBaseException(nameof(Presenter));
        internal UniTask<CampaignResult> RequestCampaignAsync(string _) => throw new NotImplForMagifyBaseException(nameof(RequestCampaignAsync));

        #endregion
    }
}
#endif