using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public class ProductsObtainer
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly PopupsHandler _popups;

        private readonly List<IProductObtainer> _embeddedObtainers = new();
        private readonly List<IProductObtainer> _externalObtainers = new();

        private readonly Subject<ProductDef> _onProductObtainBegan = new();
        private readonly Subject<ProductObtainResult> _onProductObtained = new();
        private readonly Subject<ProductObtainResult> _onProductObtainFinished = new();

        public IObservable<ProductDef> OnProductObtainBegan => _onProductObtainBegan;
        public IObservable<ProductObtainResult> OnProductObtained => _onProductObtained;
        public IObservable<ProductObtainResult> OnProductObtainFinished => _onProductObtainFinished;

        public ProductsObtainer([NotNull] PopupsHandler popups)
        {
            _popups = popups;
        }

        internal void RemoveEmbeddedObtainer<T>() where T : IProductObtainer
        {
            _embeddedObtainers.RemoveAll(c => c is T);
        }

        internal void RegisterEmbeddedProductObtainer(IProductObtainer obtainer)
        {
            _embeddedObtainers.Add(obtainer);
        }

        public void RegisterCampaignHandler(IProductObtainer obtainer)
        {
            _externalObtainers.Add(obtainer);
        }

        private IProductObtainer GetObtainerFor(ProductDef product)
        {
            return _externalObtainers.FirstOrDefault(c => c.CanObtainProduct(product)) ?? _embeddedObtainers.FirstOrDefault(c => c.CanObtainProduct(product));
        }

        public bool CanObtainProduct(ProductDef product)
        {
            return GetObtainerFor(product) != null;
        }

        internal void HandleExternalObtain(ProductDef product, PurchaseInfo purchaseInfo)
        {
            var externalObtainResult = new ProductObtainResult(product, purchaseInfo);
            _onProductObtained.OnNext(externalObtainResult);
        }

        public async UniTask<ProductObtainResult> ObtainProductAsync(ProductDef product, CampaignRequest request, CancellationToken cancellationToken)
        {
            _logger.Log($"Trying to obtain product: {product.ToJson()}");
            var obtainer = GetObtainerFor(product);
            if (obtainer == null)
            {
                _logger.LogError($"Obtainer for product {product.GetType().Name} not found.");
                return new ProductObtainResult(product, request.Event, request.Campaign, ProductObtainFailReason.NoObtainer);
            }

            return obtainer.ObtainUnderSpin
                ? await _popups.DoUnderSpin(0.5f, cancellationToken, obtain)
                : await obtain(cancellationToken);

            async UniTask<ProductObtainResult> obtain(CancellationToken ct)
            {
                if (obtainer.NeedPrepareProduct(product))
                {
                    _logger.Log("Obtainer requested product fetch before obtain");
                    var prepareResult = await obtainer.PrepareProductAsync(product, ct);
                    _logger.Log($"Preparation finished with result: {prepareResult.FailReason}");
                    if (prepareResult.FailReason != null)
                    {
                        if (!request.ImpressionTracked)
                        {
                            request.TrackShowFailed(prepareResult.FailReason.ToString());
                        }
                        else
                        {
                            request.TrackObtainFailed(product, prepareResult.FailReason.ToString());
                        }
                        return new ProductObtainResult(product, request, prepareResult.FailReason.Value);
                    }
                }
                request.TrackShow();
                ProductObtainResult? result = null;
                try
                {
                    request.TrackClick(product);
                    _onProductObtainBegan.OnNext(product);
                    result = await obtainer.ObtainProductAsync(product, request, ct);
                    cancellationToken.ThrowIfCancellationRequested();
                    _logger.Log($"Product purchasing result is: {result.Value.FailReason}");
                    switch (result.Value.FailReason)
                    {
                        case null:
                            _onProductObtained.OnNext(result.Value);
                            request.TrackObtain(result.Value);
                            break;
                        case ProductObtainFailReason.UserCancelled:
                            break;
                        default:
                            request.TrackObtainFailed(product, result.Value.FailReason.ToString());
                            break;
                    }
                }
                catch (OperationCanceledException)
                {
                    throw;
                }
                catch (Exception e)
                {
                    Debug.LogException(e);
                    result ??= new ProductObtainResult(product, request, ProductObtainFailReason.Unknown);
                }
                finally
                {
                    if (result.HasValue)
                    {
                        _onProductObtainFinished.OnNext(result.Value);
                    }
                }
                return result.Value;
            }
        }
    }
}