using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using UnityEngine;

namespace Magify
{
    internal class PopupSplashInterstitialCampaignHandler : InterstitialCampaignHandler
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly INetworkStatusProvider _networkStatus;
        private readonly DelayedCampaignRegistry _delayedCampaignRegistry;
        private readonly PopupsHandler _popups;

        public PopupSplashInterstitialCampaignHandler(AdvertiserService advertiser, INetworkStatusProvider networkStatus, DelayedCampaignRegistry delayedCampaignRegistry, PopupsHandler popups)
            : base(advertiser)
        {
            _networkStatus = networkStatus;
            _delayedCampaignRegistry = delayedCampaignRegistry;
            _popups = popups;
        }

        public override bool CanHandleCampaign(CampaignRequest request)
        {
            return request.Campaign is InterstitialCampaign { Screen: PopupSplash } && Advertiser.Mediator is { IsInterVideoReady: false };
        }

        public override UniTask<CampaignResult> HandleCampaignAsync(CampaignRequest request, CancellationToken cancellationToken)
        {
            if (!_networkStatus.IsNetworkReachable)
            {
                _logger.Log("Network is not reachable - ignore campaign");
                request.TrackShowFailed("No internet");
                return UniTask.FromResult(CampaignResult.NoInternet);
            }

            _delayedCampaignRegistry.DelayCampaign(request, ShowInterCampaignDelayed(request, cancellationToken));
            return UniTask.FromResult(CampaignResult.Delayed);
        }

        private async UniTask<CampaignResult> ShowInterCampaignDelayed(CampaignRequest request, CancellationToken cancellationToken)
        {
            var campaign = (InterstitialCampaign)request.Campaign;
            var popupSplash = (PopupSplash)campaign.Screen;

            _logger.Log($"Load InterVideo and wait for result for {request.Event} ({campaign.Name}). Timeout - {popupSplash.Timeout}");
            var gameTime = Time.realtimeSinceStartup;
            var loadResult = await Advertiser.LoadInterVideoAsync((float)popupSplash.Timeout, cancellationToken);
            cancellationToken.ThrowIfCancellationRequested(); // TODO maybe should rethrow from catch in LoadInterVideoAsync

            if (loadResult.State != AdLoadState.Loaded)
            {
                _logger.Log($"Failed to load InterVideo for {request.Event}. FailType={loadResult.State}. Error: {loadResult.Error}");
                request.TrackShowFailed($"FailType={loadResult.State}. Error: {loadResult.Error}");
                return CampaignResult.Failed;
            }

            var loadTime = Time.realtimeSinceStartup - gameTime;
            _logger.Log($"Interstitial ad loaded in {loadTime} seconds.");

            if (popupSplash.IsForced || loadTime > popupSplash.MaxTime)
            {
                try
                {
                    _logger.Log("Show 'Take a Break' popup before InterVideo");
                    await _popups.ShowInterPopupSplash(TimeSpan.FromSeconds(popupSplash.Time), cancellationToken);
                }
                catch (OperationCanceledException)
                {
                    _logger.Log("Inter loading was canceled");
                    return CampaignResult.Aborted;
                }
            }

            return await ShowIterVideoAd(request);
        }
    }
}