using System.Threading;
using Cysharp.Threading.Tasks;
using UnityEngine;

namespace Magify
{
    internal class NoSplashInterstitialCampaignHandler : InterstitialCampaignHandler
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly INetworkStatusProvider _networkStatus;
        private readonly DelayedCampaignRegistry _delayedCampaignRegistry;

        public NoSplashInterstitialCampaignHandler(AdvertiserService advertiser, INetworkStatusProvider networkStatus, DelayedCampaignRegistry delayedCampaignRegistry)
            : base(advertiser)
        {
            _networkStatus = networkStatus;
            _delayedCampaignRegistry = delayedCampaignRegistry;
        }

        public override bool CanHandleCampaign(CampaignRequest request)
        {
            return request.Campaign is InterstitialCampaign { Screen: NoSplash } && Advertiser.Mediator is { IsInterVideoReady: false };
        }

        public override UniTask<CampaignResult> HandleCampaignAsync(CampaignRequest request, CancellationToken cancellationToken)
        {
            if (!_networkStatus.IsNetworkReachable)
            {
                _logger.Log("Network is not reachable - ignore campaign");
                request.TrackShowFailed("No internet");
                return UniTask.FromResult(CampaignResult.NoInternet);
            }

            _delayedCampaignRegistry.DelayCampaign(request, ShowInterCampaignDelayed(request, cancellationToken));
            return UniTask.FromResult(CampaignResult.Delayed);
        }

        private async UniTask<CampaignResult> ShowInterCampaignDelayed(CampaignRequest request, CancellationToken cancellationToken)
        {
            var campaign = (InterstitialCampaign)request.Campaign;
            var noSplash = (NoSplash)campaign.Screen;

            _logger.Log($"Load InterVideo and wait for result for {request.Event} ({campaign.Name}). Timeout - {noSplash.Timeout}");
            var gameTime = Time.realtimeSinceStartup;
            var loadResult = await Advertiser.LoadInterVideoAsync((float)noSplash.Timeout, cancellationToken);
            cancellationToken.ThrowIfCancellationRequested(); // TODO maybe should rethrow from catch in LoadInterVideoAsync

            if (loadResult.State != AdLoadState.Loaded)
            {
                _logger.Log($"Failed to load InterVideo for {request.Event}. FailType={loadResult.State}. Error: {loadResult.Error}");
                request.TrackShowFailed($"FailType={loadResult.State}. Error: {loadResult.Error}");
                return CampaignResult.Failed;
            }

            var loadTime = Time.realtimeSinceStartup - gameTime;
            _logger.Log($"Interstitial ad loaded in {loadTime} seconds.");

            return await ShowIterVideoAd(request);
        }
    }
}