using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using UnityEngine;
using static Magify.MagifyManager;

namespace Magify
{
    public class ImageCreativeCampaignHandler : ICampaignHandler
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly ProductsObtainer _productsObtainer;
        private readonly PopupsHandler _popups;

        public ImageCreativeCampaignHandler(ProductsObtainer productsObtainer, PopupsHandler popups)
        {
            _productsObtainer = productsObtainer;
            _popups = popups;
        }

        public bool CanHandleCampaign(CampaignRequest request)
        {
            return request.Campaign is ICampaignWithProducts and ICampaignWithCreative { Creative: ImageCreative };
        }

        public async UniTask<CampaignResult> HandleCampaignAsync(CampaignRequest request, CancellationToken cancellationToken)
        {
            var campaignWithCreative = (ICampaignWithCreative)request.Campaign;
            var campaignWithProducts = (ICampaignWithProducts)request.Campaign;

            var imageCreative = (ImageCreative)campaignWithCreative.Creative;
            var products = campaignWithProducts.Products;

            var product = string.IsNullOrEmpty(request.PayoutType)
                ? products.First()
                : products.First(c => c.Payout.Any(k => k.Type == request.PayoutType));

            if (!_productsObtainer.CanObtainProduct(product))
            {
                _logger.LogError($"No product obtainer for {product.GetType().Name}");
                request.TrackShowFailed($"No product obtainer for {product.GetType().Name}");
                return CampaignResult.Failed;
            }

            _logger.Log($"Show image creative: {JsonFacade.SerializeObject(imageCreative)}");

            var bought = false;
            var showResult = await _popups.ShowImagePopup(imageCreative, request.TrackShow, cancellationToken, async () =>
            {
                var obtainResult = await _productsObtainer.ObtainProductAsync(product, request, cancellationToken);
                cancellationToken.ThrowIfCancellationRequested();
                if (obtainResult.IsBought)
                {
                    bought = true;
                    return ClickHandleResult.ClosePopup;
                }
                return ClickHandleResult.NothingToDo;
            });

            if (cancellationToken.IsCancellationRequested) return CampaignResult.Aborted;
            var isPortrait = IsPortrait;
            switch (showResult)
            {
                case PopupShowResult.NoInternet:
                    _logger.LogError($"Can't download image for {request.Campaign.Name} - No internet");
                    request.TrackShowFailed("No internet");
                    return CampaignResult.NoInternet;
                case PopupShowResult.NotFound:
                    _logger.LogError($"Image url not found for {(isPortrait ? "portrait" : "landscape")} orientation");
                    request.TrackShowFailed($"Image url not found for {(isPortrait ? "portrait" : "landscape")} orientation");
                    return CampaignResult.Failed;
                case PopupShowResult.Failed:
                    _logger.LogError("Unexpected behaviour - can't download image");
                    request.TrackShowFailed("Can't download image");
                    return CampaignResult.Failed;
                case PopupShowResult.Showed when bought:
                    return CampaignResult.Applied;
                case PopupShowResult.Showed:
                    return CampaignResult.Declined;
                default:
                    _logger.LogError("Failed to show image creative");
                    request.TrackShowFailed("Failed to show image creative popup");
                    return CampaignResult.Failed;
            }
        }
    }
}