using System;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using UnityEngine;

namespace Magify
{
    public class CustomCreativeCampaignHandler : ICampaignHandler
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly ProductsObtainer _productsObtainer;
        private readonly PopupsHandler _popups;
        private readonly LimitedTimeOfferProvider _offerProvider;

        public CustomCreativeCampaignHandler(ProductsObtainer productsObtainer, PopupsHandler popups, LimitedTimeOfferProvider offerProvider)
        {
            _productsObtainer = productsObtainer;
            _popups = popups;
            _offerProvider = offerProvider;
        }

        public bool CanHandleCampaign(CampaignRequest request)
        {
            return request.Campaign is ICampaignWithProducts and ICampaignWithCreative { Creative: CustomCreative };
        }

        public async UniTask<CampaignResult> HandleCampaignAsync(CampaignRequest request, CancellationToken cancellationToken)
        {
            var campaignWithCreative = (ICampaignWithCreative)request.Campaign;
            var campaignWithProducts = (ICampaignWithProducts)request.Campaign;

            var customCreative = (CustomCreative)campaignWithCreative.Creative;
            var products = campaignWithProducts.Products;

            if (!string.IsNullOrEmpty(request.PayoutType))
            {
                products = products.Where(c => c.Payout.Any(k => k.Type == request.PayoutType)).ToList();
            }
            foreach (var product in products)
            {
                if (_productsObtainer.CanObtainProduct(product)) continue;
                _logger.LogError($"No product obtainer for {product.GetType().Name}");
                request.TrackShowFailed($"No product obtainer for {product.GetType().Name}");
                return CampaignResult.Failed;
            }

            _logger.Log($"Show custom creative: {JsonFacade.SerializeObject(customCreative)}");

            var campaignName = request.Campaign.Name;
            var offer = _offerProvider.ActiveOffers.FirstOrDefault(offer => offer.CampaignName == campaignName);
            var args = new CreativePopup.Args(request.Event, campaignName, customCreative, products, offer);
            var bought = false;
            PopupShowResult showResult;
            try
            {
                showResult = await _popups.ShowCreativePopup(request.Campaign, args, request.TrackShow, cancellationToken, async (product, closeAfterObtain) =>
                {
                    var obtainResult = await _productsObtainer.ObtainProductAsync(product, request, cancellationToken);
                    cancellationToken.ThrowIfCancellationRequested();
                    if (obtainResult.IsBought)
                    {
                        bought = true;
                        if (closeAfterObtain) return ClickHandleResult.ClosePopup;
                    }
                    return ClickHandleResult.NothingToDo;
                });
            }
            catch (OperationCanceledException)
            {
                return CampaignResult.Aborted;
            }

            if (cancellationToken.IsCancellationRequested)
                return CampaignResult.Aborted;
            switch (showResult)
            {
                case PopupShowResult.NotFound:
                    _logger.LogError($"Custom creative with id {customCreative.ScreenId} not found");
                    request.TrackShowFailed($"Custom creative with id {customCreative.ScreenId} not found");
                    return CampaignResult.Failed;
                case PopupShowResult.Failed:
                    _logger.LogError($"Can't show creative with id {customCreative.ScreenId}");
                    request.TrackShowFailed($"Can't show creative with id {customCreative.ScreenId}");
                    return CampaignResult.Failed;
                case PopupShowResult.Showed when bought:
                    return CampaignResult.Applied;
                case PopupShowResult.Showed:
                    return CampaignResult.Declined;
                default:
                    _logger.LogError($"Failed to show creative with id {customCreative.ScreenId}");
                    request.TrackShowFailed($"Failed to show creative with id {customCreative.ScreenId}");
                    return CampaignResult.Failed;
            }
        }
    }
}