using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;
using static Magify.MagifyManager;

namespace Magify
{
    public class CampaignRequest
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        [NotNull]
        public string Event { get; }

        [NotNull]
        public ICampaign Campaign { get; }

        [NotNull]
        public CompositeDisposable Disposables { get; }

        [CanBeNull]
        public Dictionary<string, object> Params { get; init; }

        [CanBeNull]
        public string PayoutType { get; init; }

        public bool ImpressionTracked { get; private set; }

        public CampaignRequest(string @event, ICampaign campaign, CompositeDisposable disposables)
        {
            Event = @event;
            Campaign = campaign;
            Disposables = disposables;
        }

        #region Tracking

        public void TrackShowFailed(string message)
        {
            if (ImpressionTracked) return;
#if MAGIFY_3_OR_NEWER
            TrackImpressionFailFor(Campaign.Name, message);
#else
            TrackImpressionFailFor(Campaign.Type, message);
#endif
            ImpressionTracked = true;
        }

        public void TrackShow()
        {
            if (ImpressionTracked) return;
#if MAGIFY_3_OR_NEWER
            TrackImpression(Campaign.Name);
#else
            TrackImpression(Campaign.Type);
#endif
            ImpressionTracked = true;
        }

        public void TrackClick(ProductDef product = null)
        {
            switch (product)
            {
                case null:
#if MAGIFY_3_OR_NEWER
                    TrackClickFor(Campaign.Name);
#else
                    TrackClickFor(Campaign.Type);
#endif
                    break;
                default:
#if MAGIFY_3_OR_NEWER
                    TrackProductClickFor(Campaign.Name, product.Id, (product as IProductDefWithPurchaseStore)?.Store);
#else
                    TrackProductClickFor(Campaign.Type, product.Id, (product as IProductDefWithPurchaseStore)?.Store);
#endif
                    break;
            }
        }

        public void TrackAdsClick(ProductDef product = null)
        {
            if (product == null)
            {
#if MAGIFY_3_OR_NEWER
                TrackAdsClickFor(Campaign.Name);
#else
                TrackAdsClickFor(Campaign.Type);
#endif
            }
            else
            {
#if MAGIFY_3_OR_NEWER
                TrackAdsProductClickFor(Campaign.Name, product.Id);
#else
                TrackAdsProductClickFor(Campaign.Type, product.Id);
#endif
            }
        }

        public void TrackAdsShow(IAdsImpression impression)
        {
#if MAGIFY_3_OR_NEWER
            TrackAdsImpression(Campaign.Name, impression);
#else
            TrackAdsImpression(Campaign.Type, impression);
#endif
        }

        public void TrackObtainFailed(ProductDef product, string reason)
        {
            // TODO based on product we probably will use different methods. Based on MID-8793 just track impressionFail for now
            switch (product)
            {
                default:
#if MAGIFY_3_OR_NEWER
                    TrackImpressionFailFor(Campaign.Name, reason);
#else
                    TrackImpressionFailFor(Campaign.Type, reason);
#endif
                    break;
            }
        }

        public void TrackObtain(ProductObtainResult result)
        {
            if (!result.IsBought)
            {
                return;
            }
            var product = result.Product;
            var purchaseInfo = result.PurchaseInfo;

            switch (product)
            {
                case InAppProduct when result.PurchaseInfo != null && result.PurchaseInfo.TrustedPurchaseRecord != null:
                    TrackTrustedInAppFor(result.PurchaseInfo.TrustedPurchaseRecord);
                    break;
                case InAppProduct when result.PurchaseInfo != null && result.PurchaseInfo.SkipVerification:
                    TrackInAppWithoutVerification(
                        product.Id,
                        purchaseInfo.Price,
                        purchaseInfo.Currency,
                        purchaseInfo.TransactionId,
                        purchaseInfo.OriginalTransactionId,
                        purchaseInfo.Store);
                    break;

                case InAppProduct when result.PurchaseInfo != null:
                    TrackInAppFor(
                        product.Id,
                        purchaseInfo.Price,
                        purchaseInfo.Currency,
                        purchaseInfo.TransactionId,
                        purchaseInfo.PurchaseToken,
                        purchaseInfo.OriginalTransactionId,
                        purchaseInfo.Receipt?.Payload,
                        purchaseInfo.Store);
                    break;
                case SubscriptionProduct:
                    _logger.Log($"SubscriptionProduct tracked by {nameof(SubscriptionService)}");
                    break;
                case BonusProduct:
                    TrackFreeBonusGranted(product.Id);
                    break;
                case CrossPromoProduct:
                case ExternalLinkProduct:
                case InternalLinkProduct:
                case InfoProduct:
                    TrackOrdinaryProductUsed(product.Id);
                    break;
                case RewardProduct:
                    TrackRewardGranted(product.Id);
                    break;
                default:
                    _logger.LogError($"Unexpected obtained product {product.GetType().Name} ({product.Id}) - can't track it correctly.");
                    break;
            }
        }

        #endregion
    }
}