using System;
using System.IO;
using UnityEditor;
using UnityEngine;
using Debug = UnityEngine.Debug;

namespace Magify.PlayerBuilder
{
    public partial class MagifyBuildConfig
    {
        [Serializable]
        public struct BundleInfo
        {
            public string Id;
            public string Note;

            public BundleInfo(string id)
            {
                Id = id;
                Note = "";
            }

            public BundleInfo(string id, string note)
            {
                Id = id;
                Note = note;
            }
        }

        private static MagifyBuildConfig _instance;

        private const string FailedToSaveError = "Failed to save" + nameof(MagifyBuildConfig) + " to " + PathToSettings + " Make sure the settings file is writable.";
        private const string PathToSettings = "ProjectSettings/" + nameof(MagifyBuildConfig) + ".asset";

        internal static MagifyBuildConfig GetInstance()
        {
            if (_instance != null) return _instance;
            var settings = CreateInstance<MagifyBuildConfig>();
            if (File.Exists(PathToSettings))
            {
                var settingsJson = File.ReadAllText(PathToSettings);
                JsonUtility.FromJsonOverwrite(settingsJson, settings);
            }

            _instance = settings;
            return settings;
        }

        internal static void Save()
        {
            if (_instance == null) return;
            var fileInfo = new FileInfo(PathToSettings);
            var settingsJson = JsonUtility.ToJson(_instance, true);
            if (fileInfo.Exists && settingsJson == File.ReadAllText(PathToSettings)) return;

            if (!AssetDatabase.IsOpenForEdit(PathToSettings) && !AssetDatabase.MakeEditable(PathToSettings))
            {
                Debug.LogWarning(FailedToSaveError);
                return;
            }

            try
            {
                if (fileInfo.Exists && fileInfo.IsReadOnly)
                {
                    fileInfo.IsReadOnly = false;
                }
                File.WriteAllText(PathToSettings, settingsJson);
            }
            catch (Exception)
            {
                Debug.LogWarning(FailedToSaveError);
            }
        }
    }
}