using System.Collections.Generic;
using UnityEditor;
using UnityEditorInternal;
using UnityEngine;

namespace Magify.PlayerBuilder
{
    public static class CrossPromoSettingsProvider
    {
        private static GUIContent _androidTab;
        private static ReorderableList _androidCustom;

        private static GUIContent _iosTab;
        private static ReorderableList _iosCustom;

        private static bool _showEmbedded;
        private static int _tab;

        [SettingsProvider]
        public static SettingsProvider CreateAttSettingsProvider()
        {
            _androidTab = EditorGUIUtility.IconContent("BuildSettings.Android.Small");
            _iosTab = EditorGUIUtility.IconContent("BuildSettings.iPhone.Small");
            _androidCustom = setupList("Android Package Name (com.app.package_id)", MagifyBuildConfig.AndroidCrossPromoPackages);
            _iosCustom = setupList("iOS App Schema (fb944723908443353)", MagifyBuildConfig.IOSCrossPromoSchemas);

            ReorderableList setupList(string header, List<MagifyBuildConfig.BundleInfo> originalList)
            {
                return new ReorderableList(originalList, typeof(string), true, true, true, true)
                {
                    drawHeaderCallback = rect =>
                    {
                        var leftRect = rect;
                        leftRect.width /= 2;
                        var rightRect = rect;
                        rightRect.xMin = leftRect.xMax;

                        EditorGUI.LabelField(leftRect, header);
                        EditorGUI.LabelField(rightRect, "Note (optional)");
                    },
                    drawElementCallback = (rect, index, _, _) =>
                    {
                        var element = originalList[index];
                        var leftRect = rect;
                        leftRect.width /= 2;
                        var rightRect = rect;
                        rightRect.xMin = leftRect.xMax;

                        element.Id = EditorGUI.TextField(leftRect, element.Id);
                        element.Note = EditorGUI.TextField(rightRect, element.Note);

                        originalList[index] = element;
                    },
                    onAddCallback = _ => { originalList.Add(new MagifyBuildConfig.BundleInfo { Id = "New Element" }); },
                    onRemoveCallback = list => { originalList.RemoveAt(list.index); }
                };
            }

            var provider = new SettingsProvider("Project/Magify/Cross Promo Apps", SettingsScope.Project)
            {
                guiHandler = (_) =>
                {
                    EditorGUI.BeginChangeCheck();
                    using (new EditorGUILayout.VerticalScope("box"))
                    {
                        DrawHeader();
                        MagifyBuildConfig.PostprocessorsCallbackOrder = EditorGUILayout.IntField("Patch Callback Order", MagifyBuildConfig.PostprocessorsCallbackOrder);
                        switch (_tab)
                        {
                            case 0:
                                MagifyBuildConfig.AutoPatchManifestOnBuild = EditorGUILayout.Toggle("Auto Patch Manifest", MagifyBuildConfig.AutoPatchManifestOnBuild);
                                _androidCustom.DoLayoutList();
                                break;
                            case 1:
                                MagifyBuildConfig.AutoPatchPlistOnBuild = EditorGUILayout.Toggle("Auto Patch Plist", MagifyBuildConfig.AutoPatchPlistOnBuild);
                                _iosCustom.DoLayoutList();
                                break;
                        }
                    }
                    if (EditorGUI.EndChangeCheck())
                    {
                        MagifyBuildConfig.Save();
                    }
                },
                footerBarGuiHandler = () =>
                {
                    EditorGUILayout.BeginHorizontal();
                    EditorGUILayout.LabelField("Magify");
                    EditorGUILayout.EndHorizontal();
                },
                keywords = new HashSet<string>(new[] { "magify", "cross", "promo" }),
            };

            return provider;
        }

        private static void DrawHeader()
        {
            using (new EditorGUILayout.HorizontalScope(EditorStyles.toolbar))
            {
                _tab = GUILayout.Toolbar(_tab, new[] { _androidTab, _iosTab }, EditorStyles.toolbarButton);
            }

            GUILayout.Space(4);
        }

        private static void DrawTab(ReorderableList custom, ReorderableList embedded)
        {
            custom.DoLayoutList();
            var storedEnabled = GUI.enabled;
            GUI.enabled = false;
            embedded.DoLayoutList();
            GUI.enabled = storedEnabled;
        }
    }
}