﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using UnityEditor;
using UnityEditor.Compilation;
using UnityEngine;
using UnityEngine.UIElements;
using Button = UnityEngine.UIElements.Button;

namespace Magify.DependenciesInstaller
{
    internal class DependenciesInstallerWindow : EditorWindow
    {
        internal const string MenuItemPath = "Magify/Dependencies Installer";
        internal const string PathToConfiguration = "Magify/DependenciesInstaller/DependencyInstallerConfiguration";
        private const string Description = "Magify SDK requires a number of dependencies to work properly. This window is a helper for automatic installation of dependencies in the project. Below you can select the missing dependencies and how to install them, or you can install them yourself.";
        private static IReadOnlyList<DependencyElementConfiguration> _elements;
        private Button _installButton;

        #region Unity events

        private async void CreateGUI()
        {
            CompilationPipeline.compilationFinished += Repaint;
            await Load();
            minSize = new(650f, 150f + _elements.Count * 35f);
            var root = rootVisualElement;
            VisualElement dependencies;
            root.Add(DrawLabelDescription());
            root.Add(DrawLabel());
            root.Add(dependencies = DrawDependencyElements());
            root.Add(DrawInstallButton());
            dependencies.RegisterCallback<GeometryChangedEvent>(_ =>
            {
                var dependencyTextWidth = dependencies.Children().Max(dependency => dependency.Q<Label>().resolvedStyle.width);
                foreach (var dependency in dependencies.Children()) dependency.Q<Label>().style.width = dependencyTextWidth;
            });
        }

        private void OnDestroy()
        {
            CompilationPipeline.compilationFinished -= Repaint;
        }

        #endregion

        #region Control methods

        [InitializeOnLoadMethod]
        private static async void Initialize()
        {
            if ((await LoadElements()).Any(element => !element.IsInstalled)) ShowWindow();
        }

        [MenuItem(MenuItemPath)]
        private static void ShowWindow()
        {
            var window = GetWindow<DependenciesInstallerWindow>();
            window.titleContent = new("Dependencies Installer  \u2764");
            window.Show();
        }

        private async Task Load()
        {
            minSize = new(650f, 150f);
            var root = rootVisualElement;
            var loading = DrawLoading();
            root.Add(loading);
            _elements = await LoadElements();
            root.Remove(loading);
        }

        private static async Task<IReadOnlyList<DependencyElementConfiguration>> LoadElements()
        {
            while (true)
            {
                var asset = Resources.Load<DependencyInstallerConfiguration>(PathToConfiguration);
                if (asset != null) return await asset.GetElements();
                await Task.Delay(100);
            }
        }

        private void Repaint(object _)
        {
            Repaint();
        }

        private async void InstallDependencies()
        {
            var prev = _installButton.pickingMode;
            _installButton.pickingMode = PickingMode.Ignore;
            var prevBg = _installButton.style.backgroundColor;
            var prevText = _installButton.text;
            _installButton.style.backgroundColor = new(Color.gray);
            _installButton.text = "Installing...";

            try
            {
                await DependenciesInstaller.Install(_elements);
            }
            catch (Exception e)
            {
                Debug.LogError($"[Dependencies Installer Error]: {e.Message}");
            }

            _installButton.pickingMode = prev;
            _installButton.style.backgroundColor = prevBg;
            _installButton.text = prevText;
            Repaint();
        }

        #endregion

        #region Drawing methods

        private Label DrawLoading()
        {
            return new("Loading...")
            {
                style =
                {
                    fontSize = 24f,
                    alignSelf = new(Align.Center),
                    marginTop = 32f,
                },
            };
        }

        private TextElement DrawLabelDescription()
        {
            return new()
            {
                text = Description,
                style =
                {
                    fontSize = 16f,
                    position = Position.Relative,
                    alignSelf = new(Align.Center),
                    marginTop = 12f,
                    marginBottom = 24f,
                    marginRight = 12f,
                    marginLeft = 12f,
                },
            };
        }

        private Label DrawLabel()
        {
            return new("Select packages for import:")
            {
                style =
                {
                    fontSize = 14f,
                    unityFontStyleAndWeight = new(FontStyle.Italic),
                    position = Position.Relative,
                    alignSelf = new(Align.Center),
                    marginBottom = 12f,
                },
            };
        }

        private ScrollView DrawDependencyElements()
        {
            return new ScrollView
            {
                style =
                {
                    alignSelf = new(Align.Stretch),
                    borderBottomLeftRadius = 8f,
                    borderBottomRightRadius = 8f,
                    borderTopRightRadius = 8f,
                    borderTopLeftRadius = 8f,
                },
            }
            .With(box => box.Add(_elements.Select(DrawDependencyElement).ToArray().With(array => array.Last().style.marginBottom = 0f)));
        }

        private Box DrawDependencyElement(DependencyElementConfiguration element)
        {
            var elementBox = new Box
            {
                style =
                {
                    paddingLeft = 16f,
                    paddingRight = 16f,
                    paddingTop = 8f,
                    paddingBottom = 8f,
                    marginBottom = 16f,
                    borderRightWidth = 8f,

                    flexDirection = new(FlexDirection.Row),
                },
            };
            elementBox.Add(new Label(element.DisplayName)
            {
                style =
                {
                    fontSize = 16f,
                    marginRight = 32f,
                },
            });
            elementBox.Add(DrawElementInfo(element));
            return elementBox;
        }

        private VisualElement DrawElementInfo(DependencyElementConfiguration element)
        {
            var infoBox = new Box
            {
                style =
                {
                    flexDirection = new(FlexDirection.Column),
                    backgroundColor = new(StyleKeyword.None),
                },
            };
            infoBox.Add(new TextElement
            {
                text = element.Description,
                style = { marginBottom = 8f },
            });
            infoBox.Add(DrawElementButtons(element));
            infoBox.Add(new Label("Read more")
            {
                style =
                {
                    color = new(new Color(0.19f, 0.55f, 0.91f, 1f)),
                    fontSize = 11f,
                },
            }
            .With(label => label.RegisterCallback<ClickEvent>(_ => Application.OpenURL(element.ReadMoreLink))));
            return infoBox;
        }

        private VisualElement DrawElementButtons(DependencyElementConfiguration element)
        {
            if (element.IsInstalled)
            {
                return new TextElement
                {
                    text = "Installed",
                    style =
                    {
                        color = new(Color.gray),
                        unityFontStyleAndWeight = new(FontStyle.Italic),
                    },
                };
            }
            var choices = new List<string>(4);
            var modes = new List<InstallationMode>(4);
            if (!string.IsNullOrWhiteSpace(element.PackageNameId))
            {
                choices.Add("By package name (OpenUPM)");
                modes.Add(InstallationMode.ByPackageName);
            }
            if (!string.IsNullOrWhiteSpace(element.GitUrlId))
            {
                choices.Add("By git url");
                modes.Add(InstallationMode.ByGitUrl);
            }
            if (!string.IsNullOrWhiteSpace(element.ManualDownload.Url))
            {
                choices.Add("Direct download");
                modes.Add(InstallationMode.ManualDownload);
            }
            choices.Add("None");
            modes.Add(InstallationMode.None);
            return new RadioButtonGroup(string.Empty, choices).With(group => group.RegisterValueChangedCallback(@event => element.InstallationMode = modes[@event.newValue]));
        }

        private Button DrawInstallButton()
        {
            return _installButton = new Button
            {
                text = "Install selected",
                style =
                {
                    fontSize = 16f,
                    unityFontStyleAndWeight = FontStyle.Bold,
                    alignSelf = Align.Center,

                    paddingTop = 8f,
                    paddingBottom = 8f,
                    paddingLeft = 16f,
                    paddingRight = 16f,

                    marginTop = 16f,
                    marginBottom = 16f,
                },
            }
            .With(button => button.clicked += InstallDependencies);
        }

        #endregion
    }
}