using System;
using System.Collections.Generic;
using System.Linq;
using TMPro;
using UnityEngine;

namespace Magify
{
    public static class BundleFontCustomizer
    {
        private static readonly HashSet<TMP_FontAsset> _allFontAssets = new();
        private static readonly Dictionary<TMP_Text, HashSet<TMP_FontAsset>> _textToFontMap = new();
        private static GameObject _activeCreative;

        private static readonly ILocalizationEditorHelper _localizationEditorHelper;

        static BundleFontCustomizer()
        {
#if UNITY_LOCALIZATION
            _localizationEditorHelper = new UnityLocalizationEditorHelper();
#else
            _localizationEditorHelper = new DefaultLocalizationEditorHelper();
#endif
        }

        public static void PrepareFonts(GameObject creativePrefab, string cultureCode, List<string> availableLocales, bool includeAllLanguages)
        {
            _localizationEditorHelper.SetCurrentLocale(cultureCode);
            _activeCreative = creativePrefab;

            var tmpTexts = creativePrefab.GetComponentsInChildren<TMP_Text>(true);

            foreach (var tmpText in tmpTexts)
            {
                var fontAssets = CollectFonts(tmpText);
                StoreToAllFontAssets(fontAssets);
                _textToFontMap.Add(tmpText, fontAssets);
            }

            ClearFonts();
            PrepareNeededGlyphs(availableLocales, includeAllLanguages);
            SetFontSettingsToBuild();
        }

        public static void RevertFontsSettings()
        {
            ClearFonts();
            _allFontAssets.Clear();
            _textToFontMap.Clear();
            _activeCreative = null;
        }

        public static void OnPrepareFontsPostprocess(GameObject editingScopePrefabContentsRoot) => _localizationEditorHelper.RemoveLocalizationScripts(editingScopePrefabContentsRoot);

        private static void SetFontSettingsToBuild()
        {
            foreach (var fontAsset in _allFontAssets)
            {
                fontAsset.atlasPopulationMode = AtlasPopulationMode.Static;
            }
        }

        private static void ClearFonts()
        {
            foreach (var fontAsset in _allFontAssets)
            {
                fontAsset.atlasPopulationMode = AtlasPopulationMode.Dynamic;
                fontAsset.ClearFontAssetData(true);
            }
        }

        private static void StoreToAllFontAssets(HashSet<TMP_FontAsset> tmpFontAssets)
        {
            foreach (var fontAsset in tmpFontAssets)
            {
                _allFontAssets.Add(fontAsset);
            }
        }

        private static HashSet<TMP_FontAsset> CollectFonts(TMP_Text tmpText)
        {
            var result = new HashSet<TMP_FontAsset>();

            if (tmpText.font == null)
            {
                Debug.LogError("Text with empty font has been found - can't continue work", tmpText);
                return result;
            }

            result.Add(tmpText.font);
            collectFallbackFonts(tmpText.font, result);

            return result;

            void collectFallbackFonts(TMP_FontAsset fontAsset, HashSet<TMP_FontAsset> fontsLookup)
            {
                foreach (var fallbackFontAsset in fontAsset.fallbackFontAssetTable.Where(c => c != null))
                {
                    if (fontsLookup.Contains(fallbackFontAsset))
                    {
                        continue;
                    }

                    fontsLookup.Add(fallbackFontAsset);
                    collectFallbackFonts(fallbackFontAsset, fontsLookup);
                }
            }
        }

        private static void PrepareNeededGlyphs(List<string> availableLocales, bool includeAllLanguages)
        {
            var isUpperOrLowerCase = false;

            if (_allFontAssets.Any())
            {
                AddCharactersToFonts(_allFontAssets, BundleCustomizerConstants.AlwaysIncludedSymbols);
            }

            BundleLocalizationSwitcherContainer container = null;
            if (includeAllLanguages)
            {
                container = _activeCreative.AddComponent<BundleLocalizationSwitcherContainer>();
            }

            foreach (var pair in _textToFontMap)
            {
                var tmp = pair.Key;
                _localizationEditorHelper.LocalizeText(tmp);

                var usedGlyphs = BundleFontUtils.RemoveHtmlTags(_localizationEditorHelper.GetCharactersForPreparing(tmp, availableLocales, includeAllLanguages));
                var fontStyle = tmp.fontStyle;

                if ((fontStyle & FontStyles.LowerCase) != 0)
                {
                    usedGlyphs = usedGlyphs.ToLower();
                    isUpperOrLowerCase = true;
                }
                else if ((fontStyle & (FontStyles.UpperCase | FontStyles.SmallCaps)) != 0)
                {
                    usedGlyphs = usedGlyphs.ToUpper();
                    isUpperOrLowerCase = true;
                }

                AddCharactersToFonts(pair.Value, usedGlyphs);

                if (includeAllLanguages)
                {
                    var localizationStrings = _localizationEditorHelper.GetLocalizationStringForPreparing(tmp, availableLocales, true);
                    var switcher = CreateLocalizationSwitcher(container, tmp, localizationStrings);
                    if (switcher != null)
                    {
                        container.LocalizationSwitchers.Add(switcher);
                    }
                }
            }

            foreach (var pair in _textToFontMap)
            {
                var characters = _localizationEditorHelper.GetLocalizedCacheCharacters(pair.Key, isUpperOrLowerCase, availableLocales, includeAllLanguages);
                AddCharactersToFonts(_allFontAssets, characters);
            }
        }

        private static void AddCharactersToFonts(HashSet<TMP_FontAsset> fonts, string characters)
        {
            foreach (var fontAsset in fonts)
            {
                if (fontAsset.HasCharacters(characters, out var missingCharactersList))
                {
                    characters = string.Empty;
                    break;
                }

                fontAsset.TryAddCharacters(string.Join(string.Empty, missingCharactersList), out var missingCharacters);
                characters = missingCharacters;
                if (missingCharacters.Length == 0)
                {
                    break;
                }
            }

            if (characters.Length > 0)
            {
                throw new Exception($"Can't build creative <color=yellow>{_activeCreative.name}</color> " +
                                    $"for localization <color=yellow>{_localizationEditorHelper.GetCurrentLocale()}</color>. " +
                                    "$Because can't find in all fonts</color>" +
                                    $" this symbols: <color=white>{string.Join(", ", characters)}</color>");
            }
        }

        private static BundleLocalizationTextSwitcher CreateLocalizationSwitcher(BundleLocalizationSwitcherContainer container, TMP_Text tmpText, Dictionary<string, string> localizationCollection)
        {
            if (localizationCollection.All(t => t.Key == CreativeBuilderConstants.DefaultTextKey))
            {
                return null;
            }
            localizationCollection.Remove(CreativeBuilderConstants.DefaultTextKey);

            var switcher = tmpText.gameObject.AddComponent<BundleLocalizationTextSwitcher>();
            foreach (var localization in localizationCollection)
            {
                switcher.Container = container;
                switcher.TMPText = tmpText;
                switcher.LocalizationTexts.Add(new BundleLocalizationTextSwitcher.LocalizationText
                {
                    LocaleKey = localization.Key,
                    LocalizedString = localization.Value
                });
            }
            return switcher;
        }
    }
}