using System.Collections.Generic;
using System.Reflection;
using FluentAssertions;
using Magify.Model;
using Magify.Rx;
using Newtonsoft.Json;
using NUnit.Framework;
using UnityEngine;
using EventType = Magify.Model.EventType;

namespace Magify.Tests
{
    [Ignore("API removed: HandleContext")]
    internal partial class MagifyManagerTests
    {
        private const string TestConfig = "JsonModelParse/test_config";

        [TestOf(typeof(MagifyManager))]
        public class CampaignFor
        {
            [SetUp, TearDown]
            public void ClearTestData()
            {
                if (!MagifyManager.Initialized)
                    MagifyManager.Initialize(GetConfig());
                MagifyManager.Reset();
                MagifyManager.ShutDown();
            }

            [Test]
            public void CampaignForWithParams_ChangeParamsDic_ImpressionEventIsRight()
            {
                // Arrange
                var dic = new Dictionary<string, object>
                {
                    { "a", 1 },
                    { "b", "b" },
                    { "campaign", "UC Sample GP LTO in-app image main_lto_1" },
                };
                MagifyManager.Initialize(GetConfig());
                var @event = default(CampaignImpressionEvent);
                var json = Resources.Load<TextAsset>(TestConfig).text;
                // var context = CampaignsContext.Deserialize(json);
                // GetClient()!.HandleContext(context, ContextKind.Default);
                using var _ = GetEventCallback()
                    .Where(tuple => tuple.Item1 is EventType.Impression)
                    .Subscribe(tuple => @event = tuple.Item2 as CampaignImpressionEvent);

                // Act
                var campaign = MagifyManager.CampaignFor("session_start", dic);
                dic.Remove("a");
                dic.Remove("b");
                MagifyManager.TrackImpression(campaign.Type);

                // Assert
                @event.Parameters.Keys.Should().Contain("a");
                @event.Parameters.Keys.Should().Contain("b");
                Debug.Log(JsonFacade.SerializeObject(@event, Formatting.Indented));
            }

            [Test]
            public void CampaignForWithOutParams_ImpressionEventIsRight()
            {
                // Arrange
                MagifyManager.Initialize(GetConfig());
                var @event = default(CampaignImpressionEvent);
                var json = Resources.Load<TextAsset>(TestConfig).text;
                // var context = CampaignsContext.Deserialize(json);
                // GetClient()!.HandleContext(context, ContextKind.Default);
                using var _ = GetEventCallback()
                    .Where(tuple => tuple.Item1 is EventType.Impression)
                    .Subscribe(tuple => @event = tuple.Item2 as CampaignImpressionEvent);

                // Act
                var campaign = MagifyManager.CampaignFor("session_start");
                MagifyManager.TrackImpression(campaign.Type);

                // Assert
                @event.Parameters.Keys.Should().BeEmpty();
                Debug.Log(JsonFacade.SerializeObject(@event, Formatting.Indented));
            }

            [Test]
            [TestCase(CampaignRequestFlags.ReplaceCampaignTypeWithString, true)]
            [TestCase(CampaignRequestFlags.None, false)]
            public void CampaignForWithCampaignTypeParams_WithFlag_ImpressionEventIsRight(CampaignRequestFlags flags, bool isReplaced)
            {
                // Arrange
                var dic = new Dictionary<string, object>
                {
                    { "ct1", CampaignType.RewardedVideo },
                    { "ct2", CampaignType.Mixed },
                    { "nct1", 1 },
                    { "nct2", "hello" },
                    { "pct2", CampaignType.Bonus.ToEnumString() },
                };
                MagifyManager.Initialize(GetConfig());
                var @event = default(CampaignImpressionEvent);
                var json = Resources.Load<TextAsset>(TestConfig).text;
                // var context = CampaignsContext.Deserialize(json);
                // GetClient()!.HandleContext(context, ContextKind.Default);
                using var _ = GetEventCallback()
                    .Where(tuple => tuple.Item1 is EventType.Impression)
                    .Subscribe(tuple => @event = tuple.Item2 as CampaignImpressionEvent);

                // Act
                var campaign = MagifyManager.CampaignFor("session_start", flags, dic);
                MagifyManager.TrackImpression(campaign.Type);

                // Assert
                dic["ct1"].Should().Be(isReplaced ? CampaignType.RewardedVideo.ToEnumString() : CampaignType.RewardedVideo);
                dic["ct2"].Should().Be(isReplaced ? CampaignType.Mixed.ToEnumString() : CampaignType.Mixed);
                dic["nct1"].Should().Be(1);
                dic["nct2"].Should().Be("hello");
                dic["pct2"].Should().Be(CampaignType.Bonus.ToEnumString());

                @event.Parameters["ct1"].Should().Be(isReplaced ? CampaignType.RewardedVideo.ToEnumString() : CampaignType.RewardedVideo);
                @event.Parameters["ct2"].Should().Be(isReplaced ? CampaignType.Mixed.ToEnumString() : CampaignType.Mixed);
                @event.Parameters["nct1"].Should().Be(1);
                @event.Parameters["nct2"].Should().Be("hello");
                @event.Parameters["pct2"].Should().Be(CampaignType.Bonus.ToEnumString());

                Debug.Log(JsonFacade.SerializeObject(@event, Formatting.Indented));
            }

            private MagifyClient GetClient()
            {
                const BindingFlags flags = BindingFlags.Default | BindingFlags.Instance | BindingFlags.Public | BindingFlags.NonPublic;
                var clientField = (MagifyManager.MagifyPlatformAPI as MagifyCSharp).GetType().GetField("_client", flags);
                var client = clientField.GetValue(MagifyManager.MagifyPlatformAPI);
                return (MagifyClient)client;
            }

            private Subject<(EventType, IAnalyticsEvent)> GetEventCallback()
            {
                const BindingFlags flags = BindingFlags.Default | BindingFlags.Instance | BindingFlags.Public | BindingFlags.NonPublic;
                var clientField = (MagifyManager.MagifyPlatformAPI as MagifyCSharp).GetType().GetField("_client", flags);
                var client = clientField.GetValue(MagifyManager.MagifyPlatformAPI);
                var analyticsField = typeof(MagifyClient).GetField("_analyticsTracker", flags);
                var analytics = analyticsField.GetValue(client);
                var callbackField = analyticsField.FieldType.GetField("_onEventSentDetailed", flags);
                var callback = callbackField.GetValue(analytics) as Subject<(EventType, IAnalyticsEvent)>;
                return callback;
            }
        }
    }
}