using System;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    public class TimeExtensionsTests
    {
        [Test]
        public void MaxOfTimeSpan_WithBothNull_ThenResultShouldBeNull()
        {
            // Arrange
            TimeSpan? first = null;
            TimeSpan? second = null;

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().BeNull();
        }

        [Test]
        public void MaxOfTimeSpan_WithFirstNull_ThenResultShouldBeSecond()
        {
            // Arrange
            TimeSpan? first = null;
            TimeSpan? second = TimeSpan.FromHours(1);

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().Be(second);
        }

        [Test]
        public void MaxOfTimeSpan_WithSecondNull_ThenResultShouldBeFirst()
        {
            // Arrange
            TimeSpan? first = TimeSpan.FromMinutes(30);
            TimeSpan? second = null;

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().Be(first);
        }

        [Test]
        public void MaxOfTimeSpan_WithFirstGreater_ThenResultShouldBeFirst()
        {
            // Arrange
            TimeSpan? first = TimeSpan.FromHours(2);
            TimeSpan? second = TimeSpan.FromMinutes(90);

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().Be(first);
        }

        [Test]
        public void MaxOfTimeSpan_WithSecondGreater_ThenResultShouldBeSecond()
        {
            // Arrange
            TimeSpan? first = TimeSpan.FromMinutes(45);
            TimeSpan? second = TimeSpan.FromHours(1);

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().Be(second);
        }

        [Test]
        public void MaxOfDateTime_WithBothNull_ThenResultShouldBeNull()
        {
            // Arrange
            DateTime? first = null;
            DateTime? second = null;

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().BeNull();
        }

        [Test]
        public void MaxOfDateTime_WithFirstNull_ThenResultShouldBeSecond()
        {
            // Arrange
            DateTime? first = null;
            DateTime? second = new DateTime(2020, 1, 1);

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().Be(second);
        }

        [Test]
        public void MaxOfDateTime_WithSecondNull_ThenResultShouldBeFirst()
        {
            // Arrange
            DateTime? first = new DateTime(2019, 12, 31);
            DateTime? second = null;

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().Be(first);
        }

        [Test]
        public void MaxOfDateTime_WithFirstGreater_ThenResultShouldBeFirst()
        {
            // Arrange
            DateTime? first = new DateTime(2021, 6, 1);
            DateTime? second = new DateTime(2020, 6, 1);

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().Be(first);
        }

        [Test]
        public void MaxOfDateTime_WithSecondGreater_ThenResultShouldBeSecond()
        {
            // Arrange
            DateTime? first = new DateTime(2020, 1, 1);
            DateTime? second = new DateTime(2020, 6, 1);

            // Act
            var result = TimeExtensions.MaxOf(first, second);

            // Assert
            result.Should().Be(second);
        }

        [Test]
        public void MinOfDateTime_WithFirstLess_ThenResultShouldBeFirst()
        {
            // Arrange
            DateTime first = new DateTime(2020, 1, 1);
            DateTime second = new DateTime(2021, 1, 1);

            // Act
            var result = TimeExtensions.MinOf(first, second);

            // Assert
            result.Should().Be(first);
        }

        [Test]
        public void MinOfDateTime_WithSecondLess_ThenResultShouldBeSecond()
        {
            // Arrange
            DateTime first = new DateTime(2021, 1, 1);
            DateTime second = new DateTime(2020, 1, 1);

            // Act
            var result = TimeExtensions.MinOf(first, second);

            // Assert
            result.Should().Be(second);
        }

        [Test]
        public void MinOfDateTime_WithBothEqual_ThenResultShouldBeFirst()
        {
            // Arrange
            DateTime first = new DateTime(2020, 1, 1);
            DateTime second = new DateTime(2020, 1, 1);

            // Act
            var result = TimeExtensions.MinOf(first, second);

            // Assert
            result.Should().Be(first);
        }

        [Test]
        public void MinOfNullableDateTime_WithBothNull_ThenResultShouldBeNull()
        {
            // Arrange
            DateTime? first = null;
            DateTime? second = null;

            // Act
            var result = TimeExtensions.MinOf(first, second);

            // Assert
            result.Should().BeNull();
        }

        [Test]
        public void MinOfNullableDateTime_WithFirstNull_ThenResultShouldBeSecond()
        {
            // Arrange
            DateTime? first = null;
            DateTime? second = new DateTime(2020, 1, 1);

            // Act
            var result = TimeExtensions.MinOf(first, second);

            // Assert
            result.Should().Be(second);
        }

        [Test]
        public void MinOfNullableDateTime_WithSecondNull_ThenResultShouldBeFirst()
        {
            // Arrange
            DateTime? first = new DateTime(2020, 1, 1);
            DateTime? second = null;

            // Act
            var result = TimeExtensions.MinOf(first, second);

            // Assert
            result.Should().Be(first);
        }

        [Test]
        public void MinOfNullableDateTime_WithBothValues_ThenResultShouldBeFirstIfLess()
        {
            // Arrange
            DateTime? first = new DateTime(2019, 12, 31);
            DateTime? second = new DateTime(2020, 1, 1);

            // Act
            var result = TimeExtensions.MinOf(first, second);

            // Assert
            result.Should().Be(first);
        }
    }
}