using System.Linq;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using FluentAssertions;
using Magify.Rx;
using NUnit.Framework;

namespace Magify.Tests
{
    internal class ReactivePropertyTests
    {
        [Test]
        public void WhenSetValueToProperty_ThenListenerNotified_AndValueOk()
        {
            // Arrange
            const string targetValue = "hello!";
            var received = default(object);
            using var property = new ReactiveProperty<object>();
            using var _ = property.Subscribe(v => received = v);

            // Act
            property.Value = targetValue;

            // Assert
            received.Should()!.Be(targetValue);
            property.Value.Should()!.Be(targetValue);
        }

        [Test]
        public void WhenPropertyCreated_ThenListenerNotifiedWithDefaultValue()
        {
            // Arrange
            var received = default(object);
            using var property = new ReactiveProperty<object>();

            // Act
            using var _ = property.Subscribe(v => received = v);

            // Assert
            received.Should()!.BeNull();
            property.Value.Should()!.BeNull();
        }

        [Test]
        [TestCase(2)]
        [TestCase(10)]
        [TestCase(100)]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public void WhenSetValueToProperty_ThenManyListenersNotified_AndValueOk(int listenersNumber)
        {
            // Arrange
            const string targetValue = "hello!";
            var received = new object[listenersNumber];
            using var property = new ReactiveProperty<object>();
            using var disposables = new CompositeDisposable(listenersNumber);
            listenersNumber.EnumerateEach().ForEach(i => property.Subscribe(v => received[i] = v).AddTo(disposables));

            // Act
            property.Value = targetValue;

            // Assert
            property.Value.Should()!.Be(targetValue);
            received.Should()!.AllBeEquivalentTo(targetValue);
        }

        [Test]
        [TestCase(2, 2)]
        [TestCase(10, 10)]
        [TestCase(100, 100)]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public void WhenSetMultipleValuesToProperty_ThenManyListenersNotified_AndValuesHaveAll_PlusFirstNull(int listenersNumber, int valuesNum)
        {
            // Arrange
            var received = new List<object>[listenersNumber];
            using var property = new ReactiveProperty<object>();
            using var disposables = new CompositeDisposable(listenersNumber);
            listenersNumber.EnumerateEach().ForEach(i => property.Subscribe(v => (received[i] ??= new List<object>()).Add(v)).AddTo(disposables));
            var rnd = new System.Random();
            var values = valuesNum.EnumerateEach().Select(_ => (object)rnd.Next()).ToArray();
            var expected = values.Prepend(null).ToArray(); // because property created with null

            // Act
            values.ForEach(v => property.Value = v);

            // Assert
            property.Value.Should()!.Be(expected.Last());
            received.ForEach(vals => vals.Should()!.ContainInOrder(expected));
        }
    }
}