using System;
using System.Linq;
using System.Diagnostics.CodeAnalysis;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Rx;
using NUnit.Framework;

namespace Magify.Tests
{
    public class ReactivePropertyAwaiterTests
    {
        [Test]
        [Timeout(1000)]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public async Task WhenSetValueToProperty_ThenListenerNotified_AndValueOk()
        {
            // Arrange
            const string targetValue = "hello!";
            var cancellationToken = Utils.GetCancellationToken();
            var received = default(object);
            using var property = new ReactiveProperty<object>();

            // Act
            var task = UniTask.Create(async _ => received = await property, cancellationToken: cancellationToken);
            property.Value = targetValue;
            try { await task; }
            catch (Exception e) { Assert.Fail(e.Message); }

            // Assert
            received.Should()!.Be(targetValue);
            property.Value.Should()!.Be(targetValue);
        }

        [Test]
        [TestCase(2)]
        [TestCase(10)]
        [TestCase(100)]
        [Timeout(1000)]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public async Task WhenSetValueToProperty_ThenManyListenersNotified_AndValueOk(int listenersNumber)
        {
            // Arrange
            const string targetValue = "hello!";
            var cancellationToken = Utils.GetCancellationToken();
            var received = new object[listenersNumber];
            using var property = new ReactiveProperty<object>();

            // Act
            var tasks = listenersNumber.EnumerateEach().Select(i => UniTask.Create(async _ => received[i] = await property, cancellationToken: cancellationToken))!.ToArray();
            property.Value = targetValue;
            try { await UniTask.WhenAll(tasks); }
            catch (Exception e) { Assert.Fail(e.Message); }

            // Assert
            property.Value.Should()!.Be(targetValue);
            received.Should()!.AllBeEquivalentTo(targetValue);
        }
    }
}