using System;
using System.Text.RegularExpressions;
using JetBrains.Annotations;
using NUnit.Framework;
using UnityEngine;
using UnityEngine.TestTools;

namespace Magify.Tests
{
    internal class MagifyLoggerTests
    {
        [SetUp]
        [TearDown]
        public void CleanUp()
        {
            MagifyLogger.Reset();
        }

        private static object[] JustLogCases =>
            new object[]
            {
                new object[] { LogType.Log, (Action<MagifyLogger, string>)((logger, str) => logger!.Log(str)) },
                new object[] { LogType.Warning, (Action<MagifyLogger, string>)((logger, str) => logger!.LogWarning(str)) },
                new object[] { LogType.Error, (Action<MagifyLogger, string>)((logger, str) => logger!.LogError(str)) },
            };

        private static object[] SendLogCases =>
            new object[]
            {
                new object[] { "Log Message", LogType.Log, (Action<MagifyLogger, string>)((logger, str) => logger!.Log(str)) },
                new object[] { "Log Warning Message", LogType.Warning, (Action<MagifyLogger, string>)((logger, str) => logger!.LogWarning(str)) },
                new object[] { "Log Error Message", LogType.Error, (Action<MagifyLogger, string>)((logger, str) => logger!.LogError(str)) },
            };

        private static object[] HasTagCases =>
            new object[]
            {
                new object[] { LoggingScope.Content, LogType.Log, (Action<MagifyLogger, string>)((logger, str) => logger!.Log(str)) },
                new object[] { LoggingScope.Lto, LogType.Warning, (Action<MagifyLogger, string>)((logger, str) => logger!.LogWarning(str)) },
                new object[] { LoggingScope.Analytics, LogType.Error, (Action<MagifyLogger, string>)((logger, str) => logger!.LogError(str)) },
            };

        private static object[] DisableLoggingCases =>
            new object[]
            {
                new object[] { false, LogType.Log, (Action<MagifyLogger, string>)((logger, str) => logger!.Log(str)) },
                new object[] { false, LogType.Warning, (Action<MagifyLogger, string>)((logger, str) => logger!.LogWarning(str)) },
                new object[] { true, LogType.Error, (Action<MagifyLogger, string>)((logger, str) => logger!.LogError(str)) },
            };

        [Test]
        [TestCaseSource(nameof(SendLogCases))]
        public void WhenSendLogMessage_ThenLogMessageMustBeSame([NotNull] string message, LogType logType, [NotNull] Action<MagifyLogger, string> log)
        {
            //Arrange
            using var _ = new TemporaryEnableMagifyLogger();
            var logger = MagifyLogger.Get();
            var logTypeText = logType.AsString();

            //Act
            log.Invoke(logger, message);

            //Assert
            LogAssert.Expect(logType, new Regex(@"\[Magify\]\[" + logTypeText + @"\]\[\d+\]\:\ " + message));
        }

        [Test]
        [TestCaseSource(nameof(JustLogCases))]
        public void WhenSetPrefixText_ThenPrefixTextMustBeSame(LogType logType, [NotNull] Action<MagifyLogger, string> log)
        {
            //Arrange
            using var _ = new TemporaryEnableMagifyLogger();
            var logger = MagifyLogger.Get();
            var logTypeText = logType.AsString();

            //Act
            MagifyLogger.CustomPrefixSource = getPrefix;
            log.Invoke(logger, "Log Message");

            //Assert
            LogAssert.Expect(logType, new Regex(@"\[Magify\]\["+ logTypeText+ @"\]\[\d+\]\[prefix\].*"));
            return;

            string getPrefix(LogType _, string tag) => "prefix";
        }

        [Test]
        [TestCaseSource(nameof(JustLogCases))]
        public void WhenSendLog_ThenExpectSameLogType(LogType logType, [NotNull] Action<MagifyLogger, string> log)
        {
            //Arrange
            using var _ = new TemporaryEnableMagifyLogger();
            var logger = MagifyLogger.Get();
            var logTypeText = logType.AsString();

            //Act
            log.Invoke(logger, "Log Message");

            //Assert
            LogAssert.Expect(logType, new Regex(@"\[Magify\]\[" + logTypeText + @"\].*"));
        }

        [Test]
        [TestCaseSource(nameof(HasTagCases))]
        public void WhenEnableTag_ThenLogHasTag([NotNull] string tag, LogType logType, [NotNull] Action<MagifyLogger, string> log)
        {
            //Arrange
            using var _ = new TemporaryEnableMagifyLogger();
            var logger = MagifyLogger.Get(tag);
            MagifyLogger.EnableTag(tag);
            var logTypeText = logType.AsString();

            //Act
            log.Invoke(logger, "Log Message");

            //Assert
            LogAssert.Expect(logType, new Regex(@"\[Magify\]\[" + tag + @"\]\[" + logTypeText + @"\].*"));
        }

        [Test]
        [TestCaseSource(nameof(JustLogCases))]
        public void WhenEnableLogging_ThenLoggingIsEnabled(LogType logType, [NotNull] Action<MagifyLogger, string> log)
        {
            //Arrange
            using var _ = new TemporaryEnableMagifyLogger();
            var logger = MagifyLogger.Get();
            var logTypeText = logType.AsString();

            //Act
            log.Invoke(logger, "Log Message");

            //Assert
            LogAssert.Expect(logType, new Regex(@"\[Magify\]\[" + logTypeText + @"\].*"));
        }

        [Test]
        [TestCaseSource(nameof(DisableLoggingCases))]
        public void WhenDisableLogging_ThenLoggingIsDisabled(bool isLogExpected, LogType logType, [NotNull] Action<MagifyLogger, string> log)
        {
            //Arrange
            using var _ = new TemporaryDisableMagifyLogger();
            var logger = MagifyLogger.Get();
            var logTypeText = logType.AsString();

            //Act
            log.Invoke(logger, "Log Message");

            //Assert
            if (isLogExpected)
            {
                LogAssert.Expect(logType, new Regex(@"\[Magify\]\[" + logTypeText + @"\].*"));
            }
            else
            {
                LogAssert.NoUnexpectedReceived();
            }
        }
    }

    public class TemporaryEnableMagifyLogger : IDisposable
    {
        private readonly bool _wasEnabled;

        public TemporaryEnableMagifyLogger()
        {
            _wasEnabled = MagifyLogger.IsLoggingEnabled;
            MagifyLogger.IsLoggingEnabled = true;
        }

        public void Dispose()
        {
            MagifyLogger.IsLoggingEnabled = _wasEnabled;
        }
    }

    public class TemporaryDisableMagifyLogger : IDisposable
    {
        private readonly bool _wasEnabled;

        public TemporaryDisableMagifyLogger()
        {
            _wasEnabled = MagifyLogger.IsLoggingEnabled;
            MagifyLogger.IsLoggingEnabled = false;
        }

        public void Dispose()
        {
            MagifyLogger.IsLoggingEnabled = _wasEnabled;
        }
    }
}