using System;
using FluentAssertions;
using Newtonsoft.Json;
using NUnit.Framework;
using UnityEngine;
using UnityEngine.TestTools;

namespace Magify.Tests.Json
{
    public class DateFormatConverterTests
    {
        class NullableDateTimeContainer
        {
            [JsonProperty("time")]
            [JsonConverter(typeof(DateFormatConverter), "HH:mm:ss")]
            public DateTime? Time { get; set; }
        }

        class DateTimeContainer
        {
            [JsonProperty("time")]
            [JsonConverter(typeof(DateFormatConverter), "HH:mm:ss")]
            public DateTime Time { get; set; }
        }

        [Test]
        public void WhenValidDateTimeSerialized_AndDeserialized_ThenDayOfWeekIsCorrect()
        {
            // Arrange
            var json = "{\"time\":\"12:15:20\"}";

            // Act
            var period = JsonFacade.DeserializeObject<DateTimeContainer>(json);

            // Assert
            period.Time.Hour.Should().Be(12);
            period.Time.Minute.Should().Be(15);
            period.Time.Second.Should().Be(20);
        }

        [Test]
        public void WhenValidNullableDateTimeSerialized_AndDeserialized_ThenDayOfWeekIsCorrect()
        {
            // Arrange
            var json = "{\"time\":\"12:15:20\"}";

            // Act
            var period = JsonFacade.DeserializeObject<NullableDateTimeContainer>(json);

            // Assert
            period.Time.Should().NotBeNull();
            period.Time!.Value.Hour.Should().Be(12);
            period.Time!.Value.Minute.Should().Be(15);
            period.Time!.Value.Second.Should().Be(20);
        }

        [Test]
        public void WhenInvalidNullableDateTimeSerialized_AndDeserialized_ThenDayOfWeekIsCorrect()
        {
            // Arrange
            var json = "{\"time\":\"1d2:25:20\"}";
            LogAssert.Expect(LogType.Exception, "FormatException: String was not recognized as a valid DateTime.");

            // Act
            var period = JsonFacade.DeserializeObject<NullableDateTimeContainer>(json);

            // Assert
            period.Time.Should().BeNull();
        }

        [Test]
        public void WhenInvalidDateTimeSerialized_AndDeserialized_ThenDayOfWeekIsCorrect()
        {
            // Arrange
            var json = "{\"time\":\"1d2:25:20\"}";

            // Act
            Assert.Throws<FormatException>(() => { JsonConvert.DeserializeObject<DateTimeContainer>(json); });
        }
    }
}