using System;
using System.Collections.Generic;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    public class ImpressionUtilsTests
    {
        [Test]
        public void GetImpressionsPerPeriod_WithNoImpressions_ThenResultShouldBeZero()
        {
            // Arrange
            var impressionTimestamps = new Dictionary<string, List<DateTime>>
            {
                { "key1", new List<DateTime>() }
            };
            var key = "key1";
            var period = TimeSpan.FromHours(1);

            // Act
            var result = ImpressionUtils.GetImpressionsPerPeriod(impressionTimestamps, key, period);

            // Assert
            result.Should().Be(0);
        }

        [Test]
        public void GetImpressionsPerPeriod_WithAllImpressionsOutsidePeriod_ThenResultShouldBeZero()
        {
            // Arrange
            var impressionTimestamps = new Dictionary<string, List<DateTime>>
            {
                {
                    "key1", new List<DateTime>
                    {
                        DateTime.UtcNow.AddHours(-2),
                        DateTime.UtcNow.AddHours(-3)
                    }
                }
            };
            var key = "key1";
            var period = TimeSpan.FromHours(1);

            // Act
            var result = ImpressionUtils.GetImpressionsPerPeriod(impressionTimestamps, key, period);

            // Assert
            result.Should().Be(0);
        }

        [Test]
        public void GetImpressionsPerPeriod_WithSomeImpressionsWithinPeriod_ThenResultShouldBeCorrect()
        {
            // Arrange
            var impressionTimestamps = new Dictionary<string, List<DateTime>>
            {
                {
                    "key1", new List<DateTime>
                    {
                        DateTime.UtcNow.AddMinutes(-30),
                        DateTime.UtcNow.AddHours(-2),
                        DateTime.UtcNow.AddMinutes(-10)
                    }
                }
            };
            var key = "key1";
            var period = TimeSpan.FromHours(1);

            // Act
            var result = ImpressionUtils.GetImpressionsPerPeriod(impressionTimestamps, key, period);

            // Assert
            result.Should().Be(2);
        }

        [Test]
        public void GetImpressionsPerPeriod_WithAllImpressionsWithinPeriod_ThenResultShouldBeTotalCount()
        {
            // Arrange
            var impressionTimestamps = new Dictionary<string, List<DateTime>>
            {
                {
                    "key1", new List<DateTime>
                    {
                        DateTime.UtcNow.AddMinutes(-10),
                        DateTime.UtcNow.AddMinutes(-20),
                        DateTime.UtcNow.AddMinutes(-30)
                    }
                }
            };
            var key = "key1";
            var period = TimeSpan.FromHours(1);

            // Act
            var result = ImpressionUtils.GetImpressionsPerPeriod(impressionTimestamps, key, period);

            // Assert
            result.Should().Be(3);
        }

        [Test]
        public void GetImpressionsPerPeriod_WithEmptyDictionary_ThenResultShouldBeZero()
        {
            // Arrange
            var impressionTimestamps = new Dictionary<string, List<DateTime>>();
            var key = "key1";
            var period = TimeSpan.FromHours(1);

            // Act
            var result = ImpressionUtils.GetImpressionsPerPeriod(impressionTimestamps, key, period);

            // Assert
            result.Should().Be(0);
        }

        [Test]
        public void GetImpressionsPerPeriod_WithMissingKey_ThenResultShouldBeZero()
        {
            // Arrange
            var impressionTimestamps = new Dictionary<string, List<DateTime>>
            {
                {
                    "key2", new List<DateTime>
                    {
                        DateTime.UtcNow.AddMinutes(-10)
                    }
                }
            };
            var key = "key1";
            var period = TimeSpan.FromHours(1);

            // Act
            var result = ImpressionUtils.GetImpressionsPerPeriod(impressionTimestamps, key, period);

            // Assert
            result.Should().Be(0);
        }
    }
}