﻿using System.Collections.Generic;
using NUnit.Framework;

namespace Magify.Tests
{
    public class CollectionExtensionsTests
    {
        [Test]
        public void EqualsInAnyOrderSameInts()
        {
            var list1 = new List<int> { 1, 2, 3, 4, 5, 6, };
            var list2 = new List<int>(list1);
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void EqualsInAnyOrderSameShuffledInts()
        {
            var list1 = new List<int> { 1, 2, 3, 4, 5, 6, };
            var list2 = new List<int> { 2, 1, 5, 6, 3, 4, };
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void EqualsInAnyOrderBalancedDuplicatedInts()
        {
            var list1 = new List<int> { 1, 2, 2, };
            var list2 = new List<int> { 2, 2, 1, };
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void EqualsInAnyOrderSameDuplicatedInt()
        {
            var list1 = new List<int> { 1, 1, 1, };
            var list2 = new List<int> { 1, 1, 1, };
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void NotEqualsInAnyOrderDisbalancedDuplicatedInts()
        {
            var list1 = new List<int> { 1, 1, 2, };
            var list2 = new List<int> { 1, 2, 2, };
            Assert.IsFalse(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void NotEqualsInAnyOrderDifferentInts()
        {
            var list1 = new List<int> { 1, 2, 3, 4, 5, 6, };
            var list2 = new List<int>(list1) { 7, };
            Assert.IsFalse(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void ReferenceEqualsInAnyOrder()
        {
            var list1 = new List<int> { 1, 2, 3, 4, 5, 6, };
            var list2 = list1;
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void NotEqualsInAnyOrderIfOneIsNull()
        {
            var list1 = new List<int>();
            var list2 = default(List<int>);
            Assert.IsFalse(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void EqualsInAnyOrderEmpty()
        {
            var list1 = new List<int>();
            var list2 = new List<int>();
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void NotEqualsInAnyOrderWithDifferentNullString()
        {
            var list1 = new List<string> { "a", "b", "c", "d", };
            var list2 = new List<string> { "a", "b", "c", null, };
            Assert.IsFalse(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void EqualsInAnyOrderWithSameNullString()
        {
            var list1 = new List<string> { "a", "b", "c", null, };
            var list2 = new List<string> { "a", "b", "c", null, };
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void EqualsInAnyOrderWithSameManyNullString()
        {
            var list1 = new List<string> { null, "a", null, "b", "c", null, null, };
            var list2 = new List<string> { null, "a", null, "b", "c", null, null, };
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void EqualsInAnyOrderWithSameShuffledNullString()
        {
            var list1 = new List<string> { "a", null, "c", "b", };
            var list2 = new List<string> { "a", "b", "c", null, };
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void EqualsInAnyOrderWithSameShuffledManyNullString()
        {
            var list1 = new List<string> { "a", null, "c", "b", null, "e", null, };
            var list2 = new List<string> { "a", "b", "c", null, null, null, "e", };
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void EqualsInAnyOrderOnlyManyNullString()
        {
            var list1 = new List<string> { null, null, null, null, null, null, null, null, };
            var list2 = new List<string> { null, null, null, null, null, null, null, null, };
            Assert.IsTrue(list1.EqualsInAnyOrder(list2));
        }

        [Test]
        public void NotEqualsInAnyOrderOnlyManyNullString()
        {
            var list1 = new List<string> { null, null, null, null, null, null, null, };
            var list2 = new List<string> { null, null, null, null, null, null, null, null, };
            Assert.IsFalse(list1.EqualsInAnyOrder(list2));
        }
    }
}