using System;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using JetBrains.Annotations;
using NUnit.Framework;
using UnityEngine.TestTools;

namespace Magify.Tests
{
    internal class UnityWebRequestBuilderTests
    {
        private static object[] DefaultCases =>
            new object[]
            {
                new object[] { new SyncUnityWebRequestBuilder() },
                new object[] { new UnityWebRequestOnMainThreadBuilder() },
            };

        private static object[] RunOnThreadPoolCases =>
            new object[]
            {
                new object[] { new SyncUnityWebRequestBuilder(), true },
                new object[] { new UnityWebRequestOnMainThreadBuilder(), false },
            };

        [Test]
        [TestCaseSource(nameof(DefaultCases))]
        public async Task WhenRequest_ThenRequestResultShouldNotBeNull([NotNull] IUnityWebRequestBuilder builder)
        {
            //Arrange
            var cts = new CancellationTokenSource();

            //Act
            var requestResult = await builder.Post(
                string.Empty,
                Array.Empty<byte>(),
                cts.Token,
                1000);

            //Assert
            requestResult.Should()!.NotBeNull();
        }

        [Test]
        [TestCaseSource(nameof(RunOnThreadPoolCases))]
        public async Task WhenRequestOnThreadPool_ThenRequestResultShouldBeSameExpectException([NotNull] IUnityWebRequestBuilder builder, bool expectException)
        {
            //Arrange
            var cts = new CancellationTokenSource();
            var exception = default(Exception);
            LogAssert.ignoreFailingMessages = true;

            //Act
            await UniTask.RunOnThreadPool(async () =>
            {
                try
                {
                    var _ = await builder.Post(
                        string.Empty,
                        Array.Empty<byte>(),
                        cts.Token,
                        1000);
                }
                catch (Exception e)
                {
                    exception = e;
                }
            }, cancellationToken: cts.Token);

            //Assert
            if (expectException)
            {
                exception.Should()!.NotBeNull();
            }
            else
            {
                exception.Should()!.BeNull();
            }
        }
    }
}