﻿using System;
using System.Threading.Tasks;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    public class SessionCounterTests
    {
        [SetUp]
        [TearDown]
        public void ClearAllData()
        {
            EditorModeTestsEnvironment.Clear();
        }

        [Test]
        public async Task WhenApplicationWentToBackground_AndReturnedToForegroundFast_ThanSessionCounterNotChanged()
        {
            // Arrange
            using var subsystems = Create(out var sessionCounter, out var platform, out var prefs);
            subsystems.ForEach<IInitializable>(i => i.Initialize());
            var sessionId = prefs.SessionId;

            // Act
            platform.GoToBackground();
            await Task.Delay(TimeSpan.FromSeconds(0.2));
            platform.ReturnToForeground();

            // Assert
            prefs.GlobalSessionCounter.Value.Should().Be(1);
            prefs.VersionSessionCounter.Should().Be(1);
            prefs.SessionId.Should().Be(sessionId);
        }

        [Test]
        public async Task WhenApplicationWentToBackground_AndReturnedToForegroundAfterInterval_ThanSessionCounterIncremented()
        {
            // Arrange
            using var subsystems = Create(out var sessionCounter, out var platform, out var prefs);
            subsystems.ForEach<IInitializable>(i => i.Initialize());
            sessionCounter.Interval.Value = TimeSpan.FromSeconds(0.1);
            var sessionId = prefs.SessionId.Value;

            // Act
            subsystems.ForEach<IBackgroundListener>(i => i.OnBackground());
            await Task.Delay(TimeSpan.FromSeconds(0.2));
            subsystems.ForEach<IForegroundListener>(i => i.OnForeground());

            // Assert
            prefs.GlobalSessionCounter.Value.Should().Be(2);
            prefs.VersionSessionCounter.Should().Be(2);
            prefs.SessionId.Value.Should().NotBe(sessionId);
        }

        private static SubsystemsCollection Create(out SessionCounter sessionCounter, out PlatformDefault platform, out GeneralPrefs prefs)
        {
            var subsystems = new SubsystemsCollection();
            platform = new PlatformDefault().AddTo(subsystems);
            prefs = GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, new AppVersionProvider()).AddTo(subsystems);
            sessionCounter = new SessionCounter(prefs, platform).AddTo(subsystems);
            return subsystems;
        }
    }
}