using System;
using System.Collections.Generic;
using System.Threading;
using System.Reflection;
using System.Linq;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using NUnit.Framework;
using FluentAssertions;
using UnityEngine;

namespace Magify.Tests
{
    internal static partial class ServerApiTests
    {
        public class ServerApiTestsPart
        {
            [NotNull, ItemNotNull]
            private static readonly List<CancellationTokenSource> _cancellationTokenSources = new();

            [SetUp]
            [TearDown]
            public void ClearAllData()
            {
                EditorModeTestsEnvironment.Clear();
                foreach (var cts in _cancellationTokenSources)
                {
                    cts.Cancel();
                    cts.Dispose();
                }
                _cancellationTokenSources.Clear();
            }

            protected static CancellationToken GetCancellationToken(int cancelDelay = 1000)
            {
                var cancellationTokenSource = new CancellationTokenSource(cancelDelay);
                _cancellationTokenSources.Add(cancellationTokenSource);
                return cancellationTokenSource.Token;
            }

            [NotNull]
            protected static SubsystemsCollection Create([NotNull] INetworkClient networkClient, [NotNull] out ServerApi serverApi, EndpointUrl? endpointUrl = null, RuntimePlatform? platformAPI = null)
            {
                endpointUrl ??= EditorModeTestsEnvironment.DefaultEndpoint;
                var subsystems = new SubsystemsCollection();
                var platform = new PlatformDefault(GetEditorDevice(platformAPI)).AddTo(subsystems);
                var appVersionProvider = new AppVersionProvider().AddTo(subsystems);
                var generalPrefs = GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, new AppVersionProvider()).AddTo(subsystems);
                var appStatePrefs = AppStatePrefs.Create(EditorModeTestsEnvironment.AppStatePrefsPath).AddTo(subsystems);
                var authConfig = new AuthorizationConfigBuilder(EditorModeTestsEnvironment.AppName, appVersionProvider, platform, generalPrefs).AddTo(subsystems);
                serverApi = new ServerApi(endpointUrl.Value, networkClient, generalPrefs, appStatePrefs, authConfig, platform).AddTo(subsystems);
                return subsystems;
            }

            private static EditorDevice GetEditorDevice(RuntimePlatform? runtimePlatform)
            {
                if (runtimePlatform == RuntimePlatform.Android)
                    return EditorDevice.GetDefaultAndroidDevice();
                if (runtimePlatform == RuntimePlatform.IPhonePlayer)
                    return EditorDevice.GetDefaultIOSDevice();
                return null;
            }

            [NotNull, ItemNotNull]
            protected static List<AdsImpressionEvent> GetEvents()
            {
                return new List<AdsImpressionEvent>()
                {
                    new AdsImpressionEvent()
                    {
                        CampaignName = "CampaignName1",
                        SessionNumber = 1,
                        EventName = "EventName1",
                    },
                    new AdsImpressionEvent()
                    {
                        CampaignName = "CampaignName2",
                        SessionNumber = 2,
                        EventName = "EventName2",
                    },
                    new AdsImpressionEvent()
                    {
                        CampaignName = "CampaignName3",
                        SessionNumber = 3,
                        EventName = "EventName3",
                    },
                    new AdsImpressionEvent()
                    {
                        CampaignName = "CampaignName4",
                        SessionNumber = 4,
                        EventName = "EventName4",
                    },
                    new AdsImpressionEvent()
                    {
                        CampaignName = "CampaignName5",
                        SessionNumber = 5,
                        EventName = "EventName5",
                    },
                };
            }

            [NotNull]
            protected static AdsImpressionEvent GetEvent()
            {
                return GetEvents()[0];
            }
        }

        internal class BaseTest : ServerApiTestsPart
        {
            [Test]
            public void WhenServerApiIsDisposed_AllRequestContextsAreDisposed()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                var systems = Create(network, out var serverApi);
                systems.InitializeAll();

                // Act
                ((IDisposable)systems).Dispose();

                // Assert
                var fields = serverApi.GetType()
                    .GetFields(BindingFlags.Default | BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance)
                    .Where(f => f.FieldType.IsGenericType && f.FieldType.GetGenericTypeDefinition() == typeof(RequestContext<>))
                    .ToArray();
                Debug.Log(fields.Length);
                fields.ForEach(f => ((ICancelable)f.GetValue(serverApi)).IsDisposed.Should().BeTrue($"{f.Name} must be disposed"));
                fields.ForEach(f => Debug.Log(f.Name));
                fields.Length.Should().BeGreaterThan(0);
            }
        }
    }
}