using System;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal static partial class ServerApiTests
    {
        public class TrustedPurchase : ServerApiTestsPart
        {
            [Test]
            public async Task WhenServerApiDisposed_ThenSendTrustedPurchaseIsCancelled()
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                var systems = Create(network, out var serverApi);
                var disposable = systems as IDisposable;
                systems.InitializeAll();
                var cancellationToken = GetCancellationToken();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        disposable.Dispose();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.SendTrustedPurchase(new(), cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenSendTrustedPurchase_WithCancelledToken_ThenGetException()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var exception = default(MagifyAuthTokenLoadingCancelledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                try
                {
                    cts.Cancel();
                    await serverApi.SendTrustedPurchase(new(), cts.Token);
                }
                catch (MagifyAuthTokenLoadingCancelledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cts.IsCancellationRequested.Should()!.BeTrue();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenSendTrustedPurchase_AndEndpointUrlIsOffline_ThenSendTrustedResultShouldBeFalse()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var sendTrustedPurchaseResult = false;
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi, EndpointUrl.Offline);
                systems.InitializeAll();

                // Act
                try
                {
                    sendTrustedPurchaseResult = await serverApi.SendTrustedPurchase(new(), GetCancellationToken());
                }
                catch (OperationCanceledException)
                {
                    Assert.Fail();
                }

                // Assert
                sendTrustedPurchaseResult.Should()!.BeFalse();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(true, true)]
            [TestCase(false, false)]
            public async Task WhenSendTrustedPurchase_AndCancellationTokenWasCancelled_ThenGetException(bool hasCanceled, bool expectedError)
            {
                // Arrange
                var cancellationToken = GetCancellationToken(hasCanceled ? 1 : 1000);
                var exception = default(OperationCanceledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    await serverApi.SendTrustedPurchase(new(), cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                if (expectedError)
                {
                    exception.Should()!.NotBeNull();
                }
                else
                {
                    exception.Should()!.BeNull();
                }
            }

            [Test]
            [TestCase(10)]
            [TestCase(50)]
            [TestCase(100)]
            [TestCase(250)]
            public async Task WhenSendTrustedPurchase_ThenSendTrustedResultShouldBeTrue(int delayMs)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(delayMs, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var sendTrustedPurchaseResult = await serverApi.SendTrustedPurchase(new(), cancellationToken);

                // Assert
                sendTrustedPurchaseResult.Should()!.BeTrue();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(3)]
            [TestCase(5)]
            [TestCase(10)]
            [TestCase(100)]
            public void WhenSendTrustedPurchase_AndGetExpiredErrorCode_ThenRequestShouldBeRetry(int expectedFailed)
            {
                // Arrange
                var cancellationToken = GetCancellationToken(100000);
                var retries = 0;
                var network = new NetworkMoq((_, message, _) =>
                {
                    if (retries < expectedFailed)
                    {
                        retries++;
                        return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse() { Error = new() { Code = ErrorCode.Expired } });
                    }

                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                using var _1 = new TemporaryIgnoreFailingLogMessagesScope();
                systems.InitializeAll();
                serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    _ = serverApi.SendTrustedPurchase(new(), cancellationToken);
                    UniTask.Delay(100, cancellationToken : cancellationToken);
                }
                catch (Exception)
                {
                    // Ignore
                }

                // Assert
                retries.Should()!.Be(expectedFailed);
            }

            [Test]
            public async Task WhenSendTrustedPurchase_AndCancelAllSendTrustedPurchaseRequests_ThenGetOperationCancelledException()
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        serverApi.CancelAllSendTrustedPurchaseRequests();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.SendTrustedPurchase(new(),  cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
            }

            [Test]
            [TestCase(3)]
            [TestCase(5)]
            [TestCase(10)]
            [TestCase(50)]
            public async Task WhenMultiplyLoadingVerifyPurchase_ThenRequestToNetworkShouldBeSameLoadingsNumber(int loadingsNumber)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var areRequestsMade = false;
                var requestCount = 0;
                var promise = new UniTaskCompletionSource();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    // ReSharper disable once AccessToModifiedClosure
                    await UniTask.WaitUntil(() => areRequestsMade, cancellationToken: token);
                    await UniTask.Delay(10, cancellationToken: token);
                    // ReSharper disable once AccessToModifiedClosure
                    if (message.Method == WebRequestMethods.SaveTrustedPurchase)
                    {
                        requestCount++;
                        promise.TrySetResult();
                    }
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    loadingsNumber.EnumerateEach().ForEach(_ => serverApi.SendTrustedPurchase(new(), cancellationToken).AsUniTask().Forget());
                    areRequestsMade = true;
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }

                await promise.Task;
                await UniTask.Yield();

                // Assert
                requestCount.Should()!.Be(loadingsNumber);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }
        }
    }
}