using System;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;
using UnityEngine;
using UnityEngine.TestTools;

namespace Magify.Tests
{
    internal static partial class ServerApiTests
    {
        public class RestoreProgress : ServerApiTestsPart
        {
            private static object[] DefaultTestCases =>
                new object[]
                {
                    new object[] { RuntimePlatform.Android },
                    new object[] { RuntimePlatform.IPhonePlayer },
                };

            private static object[] RestoreProgressResultCases =>
                new object[]
                {
                    new object[] { ErrorCode.Expired, RuntimePlatform.Android },
                    new object[] { ErrorCode.Expired, RuntimePlatform.IPhonePlayer },
                    new object[] { ErrorCode.Banned, RuntimePlatform.Android },
                    new object[] { ErrorCode.Banned, RuntimePlatform.IPhonePlayer },
                    new object[] { ErrorCode.Revoked, RuntimePlatform.Android },
                    new object[] { ErrorCode.Revoked, RuntimePlatform.IPhonePlayer },
                };

            private static object[] MultiplyLoadingsCases =>
                new object[]
                {
                    new object[] { 2, RuntimePlatform.Android },
                    new object[] { 2, RuntimePlatform.IPhonePlayer },
                    new object[] { 5, RuntimePlatform.Android },
                    new object[] { 5, RuntimePlatform.IPhonePlayer },
                    new object[] { 100, RuntimePlatform.Android },
                    new object[] { 100, RuntimePlatform.IPhonePlayer },
                };

            private static object[] CancellationTokenCancelledCases =>
                new object[]
                {
                    new object[] { true, true, RuntimePlatform.Android },
                    new object[] { true, true, RuntimePlatform.IPhonePlayer },
                    new object[] { false, false, RuntimePlatform.Android },
                    new object[] { false, false, RuntimePlatform.IPhonePlayer },
                };

            private static object[] RequestWithDelayCases =>
                new object[]
                {
                    new object[] { 10, RuntimePlatform.Android },
                    new object[] { 10, RuntimePlatform.IPhonePlayer },
                    new object[] { 40, RuntimePlatform.Android },
                    new object[] { 40, RuntimePlatform.IPhonePlayer },
                    new object[] { 200, RuntimePlatform.IPhonePlayer },
                    new object[] { 200, RuntimePlatform.IPhonePlayer },
                };

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenRestoreProgress_AndCancelAllRestoreStateRequests_ThenGetOperationCancelledException(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var moqReached = false;
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    Debug.Log("cc");
                    if (message.Method == ApplicationStateApiMethods.GetProgress(runtimePlatform))
                    {
                        Debug.Log("bb");
                        moqReached = true;
                        await UniTask.WaitWhile(() => true, cancellationToken: token);
                        Debug.Log("rr");
                    }
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.WaitUntil(() => moqReached, cancellationToken: cancellationToken);
                        Debug.Log("aa");
                        serverApi.CancelAllRestoreStateRequests();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.RestoreState(null, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(MultiplyLoadingsCases))]
            public async Task WhenMultiplyLoadingRestoreProgress_ThenRequestToNetworkShouldBeSameLoadingsNumber(int loadingsNumber, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var requestCount = 0;
                var promise = new UniTaskCompletionSource();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    if (message.Method == ApplicationStateApiMethods.GetProgress(runtimePlatform))
                    {
                        requestCount++;
                        promise.TrySetResult();
                    }
                    await UniTask.Delay(10, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    var tasks = loadingsNumber.EnumerateEach().Select(_ => serverApi.RestoreState( null, cancellationToken).AsUniTask());
                    await UniTask.WhenAll(tasks);
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }

                await promise.Task;
                await UniTask.Yield();

                // Assert
                requestCount.Should()!.Be(loadingsNumber);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenServerApiDisposed_ThenRestoreProgressIsCancelled(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                var disposable = (IDisposable)systems;
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        disposable.Dispose();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.RestoreState( null, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenRestoreProgress_WithCancelledToken_ThenGetException(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(MagifyAuthTokenLoadingCancelledException);
                var cancellationToken = GetCancellationToken(50);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    await cancellationToken.WaitUntilCanceled();
                    await serverApi.RestoreState( null, cancellationToken);
                }
                catch (MagifyAuthTokenLoadingCancelledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenRestoreProgress_AndEndpointUrlIsOffline_ThenRestoreProgressResultShouldBeFail(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, EndpointUrl.Offline, platformAPI : runtimePlatform);
                systems.InitializeAll();

                // Act
                var saveProgressResult = await serverApi.RestoreState( null, cancellationToken);

                // Assert
                saveProgressResult.Status.Should()!.Be(RestoreStateRequestResponse.StatusCode.Fail);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(CancellationTokenCancelledCases))]
            public async Task WhenRestoreProgress_AndCancellationTokenWasCancelled_ThenGetException(bool hasCanceled, bool expectedError, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken(hasCanceled ? 1 : 1000);
                var exception = default(OperationCanceledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    await serverApi.RestoreState( null, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                if (expectedError)
                {
                    exception.Should()!.NotBeNull();
                }
                else
                {
                    exception.Should()!.BeNull();
                }
            }

            [Test]
            [TestCaseSource(nameof(RequestWithDelayCases))]
            public async Task WhenRestoreProgressWithDelay_ThenResultShouldNotBeNull(int delayMs, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(delayMs, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var restoreProgressResult = await serverApi.RestoreState( null, cancellationToken);

                // Assert
                restoreProgressResult.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(RestoreProgressResultCases))]
            public async Task WhenRestoreProgress_AndGetResponseErrorResult_ThenRestoreProgressResultShouldBeNull(ErrorCode errorCode, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse() { Error = new() { Code = errorCode } });
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();
                LogAssert.ignoreFailingMessages = true;

                // Act
                var restoreProgressResult = await serverApi.RestoreState( null, cancellationToken);

                // Assert
                restoreProgressResult.Should()!.BeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }
        }
    }
}