﻿using System;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;
using UnityEngine;
using UnityEngine.TestTools;

namespace Magify.Tests
{
    internal static partial class ServerApiTests
    {
        public class Purchase : ServerApiTestsPart
        {
            private static object[] DefaultTestCases =>
                new object[]
                {
                    new object[] { RuntimePlatform.Android },
                    new object[] { RuntimePlatform.IPhonePlayer },
                };

            private static object[] PurchaseVerificationResultCodeCases =>
                new object[]
                {
                    new object[] { PurchaseVerificationResultCode.DoesntSupport, ErrorCode.AppSupportError, RuntimePlatform.Android },
                    new object[] { PurchaseVerificationResultCode.DoesntSupport, ErrorCode.AppSupportError, RuntimePlatform.IPhonePlayer },
                    new object[] { PurchaseVerificationResultCode.Invalid, ErrorCode.InvalidPurchase, RuntimePlatform.Android },
                    new object[] { PurchaseVerificationResultCode.Invalid, ErrorCode.InvalidPurchase, RuntimePlatform.IPhonePlayer },
                    new object[] { PurchaseVerificationResultCode.InvalidGoogleCredentials, ErrorCode.InvalidGoogleCredentials, RuntimePlatform.Android },
                    new object[] { PurchaseVerificationResultCode.InvalidGoogleCredentials, ErrorCode.InvalidGoogleCredentials, RuntimePlatform.IPhonePlayer },
                    new object[] { PurchaseVerificationResultCode.Fail, ErrorCode.Expired, RuntimePlatform.Android },
                    new object[] { PurchaseVerificationResultCode.Fail, ErrorCode.Expired, RuntimePlatform.IPhonePlayer },
                    new object[] { PurchaseVerificationResultCode.Fail, ErrorCode.Banned, RuntimePlatform.Android },
                    new object[] { PurchaseVerificationResultCode.Fail, ErrorCode.Banned, RuntimePlatform.IPhonePlayer },
                    new object[] { PurchaseVerificationResultCode.Fail, ErrorCode.Revoked, RuntimePlatform.Android },
                    new object[] { PurchaseVerificationResultCode.Fail, ErrorCode.Revoked, RuntimePlatform.IPhonePlayer },
                };

            private static object[] MultiplyLoadingsCases =>
                new object[]
                {
                    new object[] { 2, RuntimePlatform.Android },
                    new object[] { 2, RuntimePlatform.IPhonePlayer },
                    new object[] { 5, RuntimePlatform.Android },
                    new object[] { 5, RuntimePlatform.IPhonePlayer },
                    new object[] { 100, RuntimePlatform.Android },
                    new object[] { 100, RuntimePlatform.IPhonePlayer },
                };

            private static object[] CancellationTokenCancelledCases =>
                new object[]
                {
                    new object[] { true, true, RuntimePlatform.Android },
                    new object[] { true, true, RuntimePlatform.IPhonePlayer },
                    new object[] { false, false, RuntimePlatform.Android },
                    new object[] { false, false, RuntimePlatform.IPhonePlayer },
                };

            private static object[] RequestWithDelayCases =>
                new object[]
                {
                    new object[] { 10, RuntimePlatform.Android },
                    new object[] { 10, RuntimePlatform.IPhonePlayer },
                    new object[] { 40, RuntimePlatform.Android },
                    new object[] { 40, RuntimePlatform.IPhonePlayer },
                    new object[] { 200, RuntimePlatform.IPhonePlayer },
                    new object[] { 200, RuntimePlatform.IPhonePlayer },
                };

            private static object[] RetryAfterExpiredErrorCases =>
                new object[]
                {
                    new object[] { 3, RuntimePlatform.Android },
                    new object[] { 3, RuntimePlatform.IPhonePlayer },
                    new object[] { 5, RuntimePlatform.Android },
                    new object[] { 5, RuntimePlatform.IPhonePlayer },
                    new object[] { 10, RuntimePlatform.IPhonePlayer },
                    new object[] { 10, RuntimePlatform.IPhonePlayer },
                };

            [Test]
            [TestCaseSource(nameof(RetryAfterExpiredErrorCases))]
            public void WhenVerifyPurchase_AndGetExpiredErrorCode_ThenRequestShouldBeRetry(int expectedFailed, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken(100000);
                var retries = 0;
                var network = new NetworkMoq((_, message, _) =>
                {
                    if (retries < expectedFailed)
                    {
                        retries++;
                        return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse() { Error = new() { Code = ErrorCode.Expired } });
                    }

                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                using var _1 = new TemporaryIgnoreFailingLogMessagesScope();
                systems.InitializeAll();
                serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    _ = serverApi.VerifyPurchase(new(), false, cancellationToken);
                    UniTask.Delay(100, cancellationToken : cancellationToken);
                }
                catch (Exception)
                {
                    // Ignore
                }

                // Assert
                retries.Should()!.Be(expectedFailed);
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenVerifyPurchase_AndCancelAllVerifyPurchaseRequests_ThenGetOperationCancelledException(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        serverApi.CancelAllVerifyPurchaseRequests();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.VerifyPurchase(new PurchaseRecord(), false, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
            }

            [Test]
            [TestCaseSource(nameof(MultiplyLoadingsCases))]
            public async Task WhenMultiplyLoadingVerifyPurchase_ThenRequestToNetworkShouldBeSameLoadingsNumber(int loadingsNumber, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var areRequestsMade = false;
                var requestCount = 0;
                var promise = new UniTaskCompletionSource();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    // ReSharper disable once AccessToModifiedClosure
                    await UniTask.WaitUntil(() => areRequestsMade, cancellationToken: token);
                    await UniTask.Delay(10, cancellationToken: token);
                    // ReSharper disable once AccessToModifiedClosure
                    if (message.Method == WebRequestMethods.VerifyPurchase(runtimePlatform))
                    {
                        requestCount++;
                        promise.TrySetResult();
                    }
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    loadingsNumber.EnumerateEach().ForEach(_ => serverApi.VerifyPurchase(new(), false, cancellationToken).AsUniTask().Forget());
                    areRequestsMade = true;
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }

                await promise.Task;
                await UniTask.Yield();

                // Assert
                requestCount.Should()!.Be(loadingsNumber);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenServerApiDisposed_ThenVerifyPurchaseIsCancelled(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                var disposable = systems as IDisposable;
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        disposable.Dispose();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.VerifyPurchase(new(), false, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenVerifyPurchase_WithCancelledToken_ThenGetException(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var exception = default(MagifyAuthTokenLoadingCancelledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                try
                {
                    cts.Cancel();
                    await serverApi.VerifyPurchase(new(), false, cts.Token);
                }
                catch (MagifyAuthTokenLoadingCancelledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cts.IsCancellationRequested.Should()!.BeTrue();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenVerifyPurchase_AndEndpointUrlIsOffline_ThenVerifyResultShouldBeFail(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var purchaseVerificationResult = default(PurchaseVerificationResult);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, EndpointUrl.Offline, platformAPI : runtimePlatform);
                systems.InitializeAll();

                // Act
                try
                {
                    purchaseVerificationResult = await serverApi.VerifyPurchase(new(), false, GetCancellationToken());
                }
                catch (OperationCanceledException)
                {
                    Assert.Fail();
                }

                // Assert
                purchaseVerificationResult.Code.Should()!.Be(PurchaseVerificationResultCode.Fail);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(CancellationTokenCancelledCases))]
            public async Task WhenVerifyPurchase_AndCancellationTokenWasCancelled_ThenGetException(bool hasCanceled, bool expectedError, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken(hasCanceled ? 1 : 1000);
                var exception = default(OperationCanceledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    await serverApi.VerifyPurchase(new(), false, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                if (expectedError)
                {
                    exception.Should()!.NotBeNull();
                }
                else
                {
                    exception.Should()!.BeNull();
                }
            }

            [Test]
            [TestCaseSource(nameof(RequestWithDelayCases))]
            public async Task WhenVerifyPurchase_ThenVerifyResultShouldNotBeSuccess(int delayMs, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(delayMs, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var purchaseVerificationResult = await serverApi.VerifyPurchase(new(), true, cancellationToken);

                // Assert
                purchaseVerificationResult.Code.Should()!.Be(PurchaseVerificationResultCode.Success);
                purchaseVerificationResult.Purchase.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(PurchaseVerificationResultCodeCases))]
            public async Task WhenVerifyPurchase_AndGetResponseErrorResult_ThenPurchaseVerificationResultCodeShouldBeSame(PurchaseVerificationResultCode expectedResultCode, ErrorCode errorCode, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse() { Error = new() { Code = errorCode } });
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();
                LogAssert.ignoreFailingMessages = true;

                // Act
                var purchaseVerificationResult = await serverApi.VerifyPurchase(new(), true, cancellationToken);

                // Assert
                purchaseVerificationResult.Code.Should()!.Be(expectedResultCode);
                purchaseVerificationResult.Purchase.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }
        }
    }
}