using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;
using EventType = Magify.Model.EventType;

namespace Magify.Tests
{
    internal static partial class ServerApiTests
    {
        public class Events : ServerApiTestsPart
        {
            [Test]
            [TestCase(3)]
            [TestCase(5)]
            [TestCase(10)]
            public void WhenSendEvents_AndGetExpiredErrorCode_ThenRequestShouldBeRetry(int expectedFailed)
            {
                // Arrange
                var cancellationToken = GetCancellationToken(100000);
                var retries = 0;
                var network = new NetworkMoq((_, message, _) =>
                {
                    if (retries < expectedFailed)
                    {
                        retries++;
                        return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse() { Error = new() { Code = ErrorCode.Expired } });
                    }

                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                using var _1 = new TemporaryIgnoreFailingLogMessagesScope();
                systems.InitializeAll();
                serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    _ = serverApi.SendEvents<AdsImpressionEvent>(EventType.Click, null!, cancellationToken);
                    UniTask.Delay(100, cancellationToken : cancellationToken);
                }
                catch (Exception)
                {
                    // Ignore
                }

                // Assert
                retries.Should()!.Be(expectedFailed);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenSendEvents_AndCancelAllEventsLoadings_ThenGetOperationCancelledException()
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        serverApi.CancelAllEventsSendings();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.SendEvents<AdsImpressionEvent>(EventType.Click, null!, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(2)]
            [TestCase(5)]
            [TestCase(100)]
            public async Task WhenMultiplySendingEvents_ThenRequestToNetworkShouldBeSameLoadingsNumber(int loadingsNumber)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var requestCount = 0;
                var network = new NetworkMoq((_, message, _) =>
                {
                    // ReSharper disable once AccessToModifiedClosure
                    if (message.Method is WebRequestMethods.Store)
                    {
                        requestCount++;
                    }
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    var tasks = loadingsNumber.EnumerateEach().Select(_ => serverApi.SendEvents<AdsImpressionEvent>(EventType.Click, null!, cancellationToken).AsUniTask());
                    await UniTask.WhenAll(tasks);
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }


                // Assert
                requestCount.Should()!.Be(loadingsNumber);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenSendEvents_AndServerApiDisposed_ThenSendEventsIsCancelled()
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                var systems = Create(network, out var serverApi);
                var disposable = systems as IDisposable;
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        disposable.Dispose();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.SendEvents<AdsImpressionEvent>(EventType.Click, null!, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenSendEvents_WithCancelledToken_ThenGetException()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var exception = default(MagifyAuthTokenLoadingCancelledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                try
                {
                    cts.Cancel();
                    await serverApi.SendEvents<AdsImpressionEvent>(EventType.Click, null!, cts.Token);
                }
                catch (MagifyAuthTokenLoadingCancelledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cts.IsCancellationRequested.Should()!.BeTrue();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenSendEvents_AndEndpointUrlIsOffline_ThenEventRequestShouldBeNullFalse()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi, EndpointUrl.Offline);
                systems.InitializeAll();

                // Act
                var eventRequest = await serverApi.SendEvents<AdsImpressionEvent>(EventType.Click, null!, cancellationToken);

                // Assert
                eventRequest.Should()!.BeFalse();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(true, true)]
            [TestCase(false, false)]
            public async Task WhenSendEvents_AndCancellationTokenWasCancelled_ThenGetException(bool hasCanceled, bool expectedError)
            {
                // Arrange
                var cancellationToken = GetCancellationToken(hasCanceled ? 1 : 1000);
                var exception = default(OperationCanceledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    await serverApi.SendEvents<AdsImpressionEvent>(EventType.Click, null!, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                if (expectedError)
                {
                    exception.Should()!.NotBeNull();
                }
                else
                {
                    exception.Should()!.BeNull();
                }
            }

            [Test]
            [TestCase(10)]
            [TestCase(40)]
            [TestCase(200)]
            public async Task WhenSendEvents_ThenRequestResultShouldBeTrue(int delayMs)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(delayMs, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var eventRequest = await serverApi.SendEvents(EventType.Click, Array.Empty<AdsImpressionEvent>(), cancellationToken);

                // Assert
                eventRequest.Should()!.BeTrue();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenSendEvents_ThenRequestedEventSame()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var requestedEvent = default(AdsImpressionEvent);
                var createdEvent = GetEvent();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    NetworkMoq.TryExtractEvent(message, EventType.Click, out requestedEvent);
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                await serverApi.SendEvents(EventType.Click, GetEvents(), cancellationToken);

                // Assert
                requestedEvent.Should()!.NotBeNull();
                requestedEvent.CampaignName.Should()!.Be(createdEvent.CampaignName);
                requestedEvent.SessionNumber.Should()!.Be(createdEvent.SessionNumber);
                requestedEvent.EventName.Should()!.Be(createdEvent.EventName);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenSendEvents_ThenRequestedSameEvents()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var requestedEvents = default(List<AdsImpressionEvent>);
                var createdEvents = GetEvents();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    NetworkMoq.TryExtractEvents(message, EventType.Click, out requestedEvents);
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                await serverApi.SendEvents(EventType.Click, createdEvents, cancellationToken);

                // Assert
                requestedEvents.Should()!.NotBeNull();
                requestedEvents.Count.Should()!.Be(createdEvents.Count);
                requestedEvents
                    .Zip(createdEvents, (requested, created) => (requested, created))
                    .ForEach(pair =>
                    {
                        pair.requested.CampaignName.Should()!.Be(pair.created.CampaignName);
                        pair.requested.SessionNumber.Should()!.Be(pair.created.SessionNumber);
                        pair.requested.EventName.Should()!.Be(pair.created.EventName);
                    });
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(2)]
            [TestCase(3)]
            [TestCase(4)]
            public async Task WhenSendDifferentEvents_ThenEventsCountSameRequests(int requestedCount)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var canBeCounted = false;
                var requestedEvents = new List<AdsImpressionEvent>();
                var createdEvents = GetEvents();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    NetworkMoq.TryExtractEvent(message, EventType.Click, out AdsImpressionEvent @event);
                    if (canBeCounted)
                    {
                        requestedEvents.Add(@event);
                    }
                    canBeCounted = true;
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                for (int i = 0; i < requestedCount; i++)
                {
                    var events = new List<AdsImpressionEvent> { createdEvents[i] };
                    await serverApi.SendEvents(EventType.Click, events, cancellationToken);
                }

                // Assert
                requestedEvents.Should()!.NotBeNull();
                requestedEvents.Count.Should()!.Be(requestedCount);
                requestedEvents
                    .Zip(createdEvents, (requested, created) => (requested, created))
                    .ForEach(pair =>
                    {
                        pair.requested.CampaignName.Should()!.Be(pair.created.CampaignName);
                        pair.requested.SessionNumber.Should()!.Be(pair.created.SessionNumber);
                        pair.requested.EventName.Should()!.Be(pair.created.EventName);
                    });
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }
        }
    }
}