﻿using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using JetBrains.Annotations;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal class ContextApplicatorTests
    {
        private class ContextListener : IContextListener
        {
            [NotNull]
            public List<ContextKind> Received { get; } = new();

            public ConfigScope SuitableScope => ConfigScopeExtensions.All();

            public void UpdateContext(CampaignsContext context, ContextKind kind)
            {
                Received.Add(kind);
            }
        }

        [Test]
        public void SavedApplied_ThenDefaultApplied_ThenDownloadedApplied_OnSameThread()
        {
            // Arrange
            using var _ = Prepare(out var applicator, out var listener);

            // Act
            applicator.HandleContext(new CampaignsContext(), ContextKind.Saved);
            applicator.HandleContext(new CampaignsContext(), ContextKind.Default);
            applicator.HandleContext(new CampaignsContext(), ContextKind.Downloaded);

            // Assert
            listener.Received.Should()!.HaveElementAt(0, ContextKind.Saved);
            listener.Received.Should()!.HaveElementAt(1, ContextKind.Default);
            listener.Received.Should()!.HaveElementAt(2, ContextKind.Downloaded);
        }

        [Test]
        public async Task SavedApplied_ThenDefaultApplied_ThenDownloadedApplied_OnDifThread()
        {
            // Arrange
            using var _ = Prepare(out var applicator, out var listener);
            var finished = 0;

            // Act
            UniTask.RunOnThreadPool(() =>
            {
                applicator.HandleContext(new CampaignsContext(), ContextKind.Downloaded);
                Interlocked.Increment(ref finished);
            }).Forget();
            UniTask.RunOnThreadPool(() =>
            {
                applicator.HandleContext(new CampaignsContext(), ContextKind.Default);
                Interlocked.Increment(ref finished);
            }).Forget();
            UniTask.RunOnThreadPool(() =>
            {
                applicator.HandleContext(new CampaignsContext(), ContextKind.Saved);
                Interlocked.Increment(ref finished);
            }).Forget();

            try
            {
                var token = new CancellationTokenSource(2000).Token;
                await UniTask.WaitWhile(() => finished < 3, cancellationToken: token);
                await UniTask.Delay(10, cancellationToken: token);
                await UniTask.SwitchToMainThread();
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            // Assert
            listener.Received.Should()!.HaveElementAt(0, ContextKind.Saved);
            listener.Received.Should()!.HaveElementAt(1, ContextKind.Default);
            listener.Received.Should()!.HaveElementAt(2, ContextKind.Downloaded);
        }

        [Test]
        [TestCase(false, false)]
        [TestCase(true, false)]
        [TestCase(false, true)]
        [TestCase(true, true)]
        public async Task SavedOrDefaultIsNull_ThenDownloadedAppliedSuccessfully(bool savedNull, bool defaultNull)
        {
            // Arrange
            using var _ = Prepare(out var applicator, out var listener);

            // Act
            applicator.HandleContext(savedNull ? null : new CampaignsContext(), ContextKind.Saved);
            applicator.HandleContext(defaultNull ? null : new CampaignsContext(), ContextKind.Default);
            applicator.HandleContext(new CampaignsContext(), ContextKind.Downloaded);

            await UniTask.Delay(10);
            await UniTask.SwitchToMainThread();

            // Assert0
            if (savedNull is false)
            {
                listener.Received.Should()!.Contain(ContextKind.Saved);
            }
            if (defaultNull is false)
            {
                listener.Received.Should()!.Contain(ContextKind.Default);
            }
            listener.Received.Should()!.Contain(ContextKind.Downloaded);
        }

        [Test]
        [TestCase(3, 2, 1, 10)]
        [TestCase(10, 5, 1, 20)]
        [TestCase(100, 50, 10, 200)]
        public async Task SavedApplied_ThenDefaultApplied_ThenDownloadedApplied_OnDifThreadWithDelay(int delaySaved, int delayDefault, int delayDownloaded, int waitBeforeCheck)
        {
            // Arrange
            using var _ = Prepare(out var applicator, out var listener);
            var finished = 0;

            // Act
            UniTask.RunOnThreadPool(() =>
            {
                UniTask.Delay(delayDownloaded).ContinueWith(() =>
                {
                    applicator.HandleContext(new CampaignsContext(), ContextKind.Downloaded);
                    Interlocked.Increment(ref finished);
                });
            }).Forget();
            UniTask.RunOnThreadPool(() =>
            {
                UniTask.Delay(delayDefault).ContinueWith(() =>
                {
                    applicator.HandleContext(new CampaignsContext(), ContextKind.Default);
                    Interlocked.Increment(ref finished);
                });
            }).Forget();
            UniTask.RunOnThreadPool(() =>
            {
                UniTask.Delay(delaySaved).ContinueWith(() =>
                {
                    applicator.HandleContext(new CampaignsContext(), ContextKind.Saved);
                    Interlocked.Increment(ref finished);
                });
            }).Forget();

            try
            {
                var token = new CancellationTokenSource(2000).Token;
                await UniTask.WaitWhile(() => finished < 3, cancellationToken: token);
                await UniTask.Delay(waitBeforeCheck, cancellationToken: token);
                await UniTask.SwitchToMainThread();
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            // Assert
            listener.Received.Should()!.HaveElementAt(0, ContextKind.Saved);
            listener.Received.Should()!.HaveElementAt(1, ContextKind.Default);
            listener.Received.Should()!.HaveElementAt(2, ContextKind.Downloaded);
        }

        [NotNull]
        private static IDisposable Prepare([NotNull] out ContextApplicator applicator, [NotNull] out ContextListener listener)
        {
            var subsystems = new SubsystemsCollection();
            applicator = new ContextApplicator(subsystems).AddTo(subsystems)!;
            listener = new ContextListener().AddTo(subsystems)!;
            return subsystems;
        }
    }
}