using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using FluentAssertions;
using Magify.Model;
using Magify.Rx;
using NUnit.Framework;
using UnityEngine.Pool;
using static Magify.Tests.Utils.CampaignTrackingUtils;

namespace Magify.Tests
{
    internal class CountersChangingScope<T> : IDisposable
        where T : unmanaged, Enum
    {
        private readonly Counter<T> _counter;
        private readonly IEnumerable<T> _scopes;
        private readonly Dictionary<T, int> _snapshot;
        private readonly CounterKey _key;
        private readonly int _offset;
        private readonly T[] _ignored;

        public CountersChangingScope(Counter<T> counter, IEnumerable<T> scopes, Dictionary<T, int> snapshot, CounterKey key, int offset, params T[] ignored)
        {
            _counter = counter;
            _scopes = scopes;
            _snapshot = snapshot;
            _key = key;
            _offset = offset;
            _ignored = ignored;
        }

        public void Dispose()
        {
            foreach (var scope in _scopes.Where(s => !_ignored.Contains(s)))
            {
                if (_counter[scope, _key] != _snapshot[scope] + _offset)
                {
                    throw new Exception("Found unexpected counter value:" +
                                        $"\nScope: {scope.ToString()}" +
                                        $"\nKey: {_key}" +
                                        $"\nCounter value: {_counter[scope, _key]}" +
                                        $"\nExpected: {_snapshot[scope] + _offset} (snapshot value: {_snapshot[scope]} + offset: {_offset})");
                }
            }
        }
    }

    [TestOf(typeof(CampaignsTracker))]
    public class CampaignsTrackerTest
    {
        private const string CampaignName = "testCampaignName";
        private const string ProductId = "testProductId";
        private const string NestedName = "testProductName";
        private const string EventSource = "tests";

        [SetUp]
        [TearDown]
        public void ClearAllData()
        {
            EditorModeTestsEnvironment.Clear();
        }

        #region Getters

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetCampaignRequest))]
        public void TrackCampaignRequest_ThenGetCampaignRequest()
        {
            // Arrange
            using (CreateCampaignsTracker(out var campaignsTracker))
            {
                CreateAndTrackRequest<BannerCampaign>(CampaignName, campaignsTracker);

                // Act
                var request = campaignsTracker.GetCampaignRequest(CampaignName);

                // Assert
                Assert.IsNotNull(request);
            }
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetCampaignRequest))]
        public void TrackCampaignRequest_ThenGetCampaignRequestByType()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackRequest<BannerCampaign>(CampaignName, campaignsTracker);

            // Act
            var request = campaignsTracker.GetCampaignRequest(CampaignType.Banner);

            // Assert
            Assert.IsNotNull(request);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetNestedForCampaign))]
        public void TrackCampaignRequestAndTrackClick_ThenGetNestedForCampaign()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackClick<BannerCampaign>(CampaignName, ProductId, NestedName, campaignsTracker);

            // Act
            var nestedName = campaignsTracker.GetNestedForCampaign(CampaignName);

            // Assert
            Assert.IsTrue(nestedName is NestedName);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetCampaignRequestForProduct))]
        public void TrackCampaignRequestAndTrackClick_ThenGetCampaignRequestForProduct()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackClick<BannerCampaign>(CampaignName, ProductId, NestedName, campaignsTracker);

            // Act
            var request = campaignsTracker.GetCampaignRequestForProduct(ProductId);

            // Assert
            Assert.IsTrue(
                request != null &&
                request.Campaign.Name == CampaignName &&
                request.NestedCampaigns!.Any(i => i.Name == NestedName && i.ProductId == ProductId));
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.CreateCampaignImpression))]
        public void TrackCampaignRequest_ThenGetCampaignImpression()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackRequest<BannerCampaign>(CampaignName, campaignsTracker);

            // Act
            var request = campaignsTracker.CreateCampaignImpression(CampaignName);

            // Assert
            Assert.IsNotNull(request);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetNestedCampaignImpression))]
        public void TrackCampaignRequestAndTrackClick_ThenGetNestedCampaignImpression()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackClick<BannerCampaign>(CampaignName, ProductId, NestedName, campaignsTracker);

            // Act
            var impression = campaignsTracker.GetNestedCampaignImpression(ProductId);

            // Assert
            Assert.IsTrue(impression is { CampaignName: NestedName });
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetNestedAdsCampaignImpression))]
        public void TrackCampaignRequestAndTrackClick_ThenGetNestedAdsCampaignImpression()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackClick<RewardedVideoCampaign>(CampaignName, ProductId, NestedName, campaignsTracker);

            // Act
            var impression = campaignsTracker.GetNestedAdsCampaignImpression(CampaignName);

            // Assert
            Assert.IsTrue(impression is { CampaignName: NestedName });
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetNestedCampaignImpressions))]
        public void TrackCampaignRequestAndTrackClick_ThenGetNestedCampaignImpressions()
        {
            // Arrange
            var productNames = new List<string>
            {
                "testProductName1",
                "testProductName2"
            };
            var productIds = new List<string>
            {
                "testProductId1",
                "testProductId2"
            };
            var nesteds = new List<NestedCampaign>
            {
                new()
                {
                    ProductId = "testProductId1",
                    Name = "testProductName1"
                },
                new()
                {
                    ProductId = "testProductId2",
                    Name = "testProductName2"
                }
            };
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackClick<RewardedVideoCampaign>(CampaignName, nesteds, campaignsTracker);

            // Act
            var impression = campaignsTracker.GetNestedCampaignImpressions(CampaignName, productIds);

            // Assert
            var equals = productNames.EqualsInAnyOrder(impression.Select(n => n.CampaignName).ToList());
            Assert.IsTrue(equals);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetNestedCampaignImpressions))]
        [TestCase(1)]
        [TestCase(3)]
        [TestCase(100)]
        public void TrackCampaignRequestAndTrackClick_ThenGetNestedCampaignImpressions_AndCheckValidity(int impressionsNumber)
        {
            // Arrange
            const string productId = "testProductId1";
            const string productIdCreative = "testProductIdCreative1";
            var productIds = new List<string>
            {
                productId,
            };
            var nesteds = new List<NestedCampaign>
            {
                new()
                {
                    ProductId = productId,
                    Name = "testProductName1",
                    ProductIdCreative = productIdCreative,
                    StoreName = PurchaseStore.Native,
                },
            };
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            for (var i = 0; i < impressionsNumber; i++)
            {
                CreateAndTrackImpression<RewardedVideoCampaign>(CampaignName, campaignsTracker);
            }
            CreateAndTrackClick<RewardedVideoCampaign>(CampaignName, nesteds, campaignsTracker);
            var parentImpression = campaignsTracker.CreateCampaignImpression(CampaignName);

            // Act
            var impressions = campaignsTracker.GetNestedCampaignImpressions(CampaignName, productIds);

            // Assert
            parentImpression.Should().NotBeNull();
            parentImpression!.CampaignName.Should().Be(CampaignName);
            impressions.Should().NotBeNullOrEmpty();
            impressions
                .Select(i => i.CampaignName).ToArray()
                .ShouldBeSameInAnyOrder(nesteds.Select(n => n.Name).ToArray());

            const BindingFlags flags = BindingFlags.Public | BindingFlags.Instance;
            var type = parentImpression!.GetType();
            var fields = type.GetFields(flags)
                .Where(m => m.Name != nameof(CampaignImpression.CampaignName) && m.Name != nameof(CampaignImpression.ProductId) && m.Name != nameof(CampaignImpression.ProductIdCreative))
                .ToArray();
            var properties = type.GetProperties(flags)
                .Where(m =>
                    m.Name != nameof(CampaignImpression.CampaignName)
                 && m.Name != nameof(CampaignImpression.ProductId)
                 && m.Name != nameof(CampaignImpression.ProductIdCreative)
                 && m.Name != nameof(CampaignImpression.StoreName))
                .ToArray();
            foreach (var nestedImpression in impressions)
            {
                nestedImpression.ProductId.Should().Be(productId);
                nestedImpression.ProductIdCreative.Should().Be(productIdCreative);
                fields.ForEach(f =>
                    f.GetValue(parentImpression)
                    .Should().Be(
                    f.GetValue(nestedImpression)));
                properties.ForEach(p =>
                    p.GetValue(parentImpression)
                    .Should().Be(
                    p.GetValue(nestedImpression)));
            }
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.BuildExternalInAppImpression))]
        public void BuildExternalInAppImpression()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);

            // Act
            var impression = campaignsTracker.BuildExternalInAppImpression(ProductId);

            // Assert
            Assert.IsTrue(impression is { CampaignName: not null, EventName: not null, ProductId: not null });
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.BuildExternalSubscriptionImpression))]
        public void BuildExternalSubscriptionImpression()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);

            // Act
            var impression = campaignsTracker.BuildExternalSubscriptionImpression(ProductId);

            // Assert
            Assert.IsTrue(impression is { CampaignName: not null, EventName: not null, ProductId: not null });
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetLastImpressionTimestampByLimitedSource))]
        public void TrackCampaignRequestAndTrackImpression_ThenGetLastImpressionTimestampByLimitedSource()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackImpression<BannerCampaign>(CampaignName, campaignsTracker);

            // Act
            var time = campaignsTracker.GetLastImpressionTimestampByLimitedSource(CampaignType.Banner);

            // Assert
            Assert.IsTrue(time != null);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetLastAdsImpressionTimestamp))]
        public void TrackCampaignRequestAndTrackImpression_ThenGetLastAdsImpressionTimestamp()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackAdsImpression<RewardedVideoCampaign>(CampaignName, campaignsTracker);

            // Act
            var time = campaignsTracker.GetLastAdsImpressionTimestamp(new[] { CampaignType.RewardedVideo });

            // Assert
            Assert.IsTrue(time != null);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.GetRelativeImpressionTimestamp))]
        public void TrackCampaignRequestAndTrackImpression_ThenGetRelativeImpressionTimestamp()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            CreateAndTrackAdsImpression<RewardedVideoCampaign>(CampaignName, campaignsTracker);

            // Act
            var time = campaignsTracker.GetRelativeImpressionTimestamp(CampaignType.Interstitial);

            // Assert
            Assert.IsTrue(time != null);
        }

        #endregion

        #region Tracks

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.RegisterCampaignRequest))]
        public void TrackCampaignRequest()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            var request = new CampaignRequest()
            {
                Campaign = new BannerCampaign()
                {
                    Name = CampaignName
                }
            };

            // Act
            campaignsTracker.RegisterCampaignRequest(request);

            // Assert
            Assert.Pass();
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.TrackImpression))]
        public void TrackCampaignRequest_AndTrackImpression()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            var request = new CampaignRequest()
            {
                Campaign = new BannerCampaign()
                {
                    Name = CampaignName
                },
                Source = new CampaignSource(EventSource, SourceType.Event)
            };
            campaignsTracker.RegisterCampaignRequest(request);

            // Act
            campaignsTracker.TrackImpression(CampaignName);

            // Assert
            Assert.Pass();
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.TrackAdsImpression))]
        public void TrackCampaignRequest_AndTrackAdsImpression()
        {
            // Arrange
            using var _ = CreateCampaignsTracker(out var campaignsTracker);
            var request = new CampaignRequest
            {
                Campaign = new RewardedVideoCampaign
                {
                    Name = CampaignName
                },
                Source = new CampaignSource(EventSource, SourceType.Event)
            };
            campaignsTracker.RegisterCampaignRequest(request);

            // Act
            campaignsTracker.TrackAdsImpression(CampaignName);

            // Assert
            Assert.Pass();
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.TrackEvent))]
        public void WhenTrackEvent_ThenCountersChanged()
        {
            // Arrange
            var keyEvent = CounterKey.GetKey(source: EventSource);
            using var _1 = CreateCampaignsTracker(out var campaignsTracker, out var counters, out _);
            using var _2 = CreateCountersChangingScope(counters.Events, keyEvent, 1);

            // Act
            campaignsTracker.TrackEvent(EventSource);

            // Assert
            Assert.Pass();
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.RevertEvent))]
        public void WhenRevertEvent_ThenCountersChanged()
        {
            // Arrange
            var keySource = CounterKey.GetKey(source: CampaignName);
            using var _1 = CreateCampaignsTracker(out var campaignsTracker, out var counters, out _);
            campaignsTracker.TrackEvent(CampaignName);
            using var _2 = CreateCountersChangingScope(counters.Events, keySource, -1);

            // Act
            campaignsTracker.RevertEvent(CampaignName);

            // Assert
            Assert.Pass();
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.TrackClick))]
        public void WhenTrackClick_ThenCountersChanged()
        {
            // Arrange
            var keyCampaignName = CounterKey.GetKey(campaignName: CampaignName);
            using var _1 = CreateCampaignsTracker(out var campaignsTracker, out var counters, out _);
            CreateAndTrackRequest<BannerCampaign>(CampaignName, campaignsTracker);
            using var _2 = CreateCountersChangingScope(counters.Clicks, keyCampaignName, 1);

            // Act
            campaignsTracker.TrackClick(CampaignName);

            // Assert
            Assert.Pass();
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.TrackInAppPurchase))]
        public void WhenTrackInAppPurchase_ThenCountersChanged_InAppStatusChanged_PurchasedInAppProductsChanged()
        {
            // Arrange
            using var _1 = CreateCampaignsTracker(out var campaignsTracker, out var counters, out var generalPrefs);
            var onPurchasedProductsChangedInvoked = false;
            var keyNestedName = CounterKey.GetKey(nestedName: NestedName);
            CreateAndTrackClick<RewardedVideoCampaign>(CampaignName, ProductId, NestedName, campaignsTracker);
            campaignsTracker.OnPurchasedProductsChanged += () => onPurchasedProductsChangedInvoked = true;
            using var _2 = CreateCountersChangingScope(counters.Nested, keyNestedName, 1, NestedCounterScope.Activation);

            // Act
            campaignsTracker.TrackInAppPurchase(ProductId, InAppSourceKind.Internal);

            // Assert
            onPurchasedProductsChangedInvoked.Should().Be(true);
            generalPrefs.InAppStatus.Value.Should().Be(InAppStatus.Purchased);
            generalPrefs.PurchasedInAppProducts.Should().Contain(ProductId);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.TrackSubscriptionPurchase))]
        public void WhenTrackSubscriptionPurchase_ThenCountersChanged_PurchasedSubscriptionProductsChanged()
        {
            // Arrange
            using var _1 = CreateCampaignsTracker(out var campaignsTracker, out var counters, out var generalPrefs);
            var onPurchasedProductsChangedInvoked = false;
            var keyNestedName = CounterKey.GetKey(nestedName: NestedName);
            CreateAndTrackClick<RewardedVideoCampaign>(CampaignName, ProductId, NestedName, campaignsTracker);
            campaignsTracker.OnPurchasedProductsChanged += () => onPurchasedProductsChangedInvoked = true;
            using var _2 = CreateCountersChangingScope(counters.Nested, keyNestedName, 1, NestedCounterScope.Activation);

            // Act
            campaignsTracker.TrackSubscriptionPurchase(ProductId, InAppSourceKind.Internal);

            // Assert
            onPurchasedProductsChangedInvoked.Should().Be(true);
            generalPrefs.PurchasedSubscriptionProducts.Should().Contain(ProductId);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.TrackRewardGranted))]
        public void WhenTrackRewardGranted_ThenCountersChanged_UsedFreeProductsChanged()
        {
            // Arrange
            var keyCampaignName = CounterKey.GetKey(campaignName: CampaignName);
            var keyNestedName = CounterKey.GetKey(nestedName: NestedName);

            using var _1 = CreateCampaignsTracker(out var campaignsTracker, out var counters, out var generalPrefs);
            CreateAndTrackClick<RewardedVideoCampaign>(CampaignName, ProductId, NestedName, campaignsTracker);
            using var _2 = CreateCountersChangingScope(counters.Rewards, keyCampaignName, 1, RewardsCounterScope.Activation);
            using var _3 = CreateCountersChangingScope(counters.Nested, keyNestedName, 1, NestedCounterScope.Activation);

            // Act
            campaignsTracker.TrackRewardGranted(ProductId);

            // Assert
            generalPrefs.UsedFreeProducts.Should().Contain(ProductId);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.TrackBonusGranted))]
        public void WhenTrackBonusGranted_ThenCountersChanged_UsedFreeProductsChanged()
        {
            // Arrange
            var keyCampaignName = CounterKey.GetKey(campaignName: CampaignName);
            using var _1 = CreateCampaignsTracker(out var campaignsTracker, out var counters, out var generalPrefs);
            CreateAndTrackClick<RewardedVideoCampaign>(CampaignName, ProductId, NestedName, campaignsTracker);
            using var _2 = CreateCountersChangingScope(counters.Bonuses, keyCampaignName, 1);

            // Act
            campaignsTracker.TrackBonusGranted(ProductId);

            // Assert
            generalPrefs.UsedFreeProducts.Should().Contain(ProductId);
        }

        [Test]
        [UnitTestTarget(typeof(CampaignsTracker), nameof(CampaignsTracker.TrackOrdinaryProductUsed))]
        public void WhenTrackOrdinaryProductUsed_ThenCountersChanged()
        {
            // Arrange
            var keyNestedName = CounterKey.GetKey(nestedName: NestedName);
            using var _1 = CreateCampaignsTracker(out var campaignsTracker, out var counters, out var generalPrefs);
            CreateAndTrackClick<RewardedVideoCampaign>(CampaignName, ProductId, NestedName, campaignsTracker);
            using var _2 = CreateCountersChangingScope(counters.Nested, keyNestedName, 1, NestedCounterScope.Activation);

            // Act
            campaignsTracker.TrackOrdinaryProductUsed(ProductId);

            // Assert
            generalPrefs.UsedFreeProducts.Should().Contain(ProductId);
        }

        #endregion

        #region Helpers

        private IDisposable CreateCampaignsTracker(out CampaignsTracker campaignsTracker, out Counters counters, out GeneralPrefs generalPrefs)
        {
            var disposables = GenericPool<CompositeDisposable>.Get();
            var platform = new PlatformDefault().AddTo(disposables);
            generalPrefs = CreateGeneralPrefs(platform).AddTo(disposables);
            generalPrefs.Reset();
            var countersStorage = new CountersStorage(EditorModeTestsEnvironment.CountersFolderPath).AddTo(disposables);
            counters = new Counters(countersStorage);
            var sessionCounter = new SessionCounter(generalPrefs, platform).AddTo(disposables);
            var limitsHolder = new LimitsHolder(generalPrefs, sessionCounter);
            campaignsTracker = new CampaignsTracker(generalPrefs, counters, limitsHolder, platform).AddTo(disposables);
            return disposables;
        }

        private IDisposable CreateCampaignsTracker(out CampaignsTracker campaignsTracker)
        {
            return CreateCampaignsTracker(out campaignsTracker, out _, out _);
        }

        private static GeneralPrefs CreateGeneralPrefs(PlatformAPI platform)
        {
            return GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, new AppVersionProvider());
        }

        #endregion
    }
}