using System.Collections.Generic;
using FluentAssertions;
using Magify.Model;
using Magify.Rx;
using Magify.Tests.Utils;
using NUnit.Framework;
using EventType = Magify.Model.EventType;

namespace Magify.Tests
{
    internal partial class MagifyClientTests
    {
        public class InApp : MagifyClientTestsPart
        {
            private const string Campaign = "campaign";
            private const string Product1 = "product1";
            private const string Product1Name = "product1Name";
            private const string Product2 = "product2";
            private const string Product2Name = "product2Name";
            private const string TransactionSource = "testTransactions";
            private const string Bonus1 = "bonus1";

            [Test]
            [UnitTestTarget(typeof(MagifyClient), nameof(MagifyClient.TrackInAppFor))]
            public void WhenTrackFirstInApp_ThenUserIsFree_WhenTrackNextInApp_ThenUserIsPaying()
            {
                // Arrange
                using var noFailOnError = new TemporaryIgnoreFailingLogMessagesScope();

                using var client = CreateDefaultAndInit();
                var campaignsTracker = GetCampaignsTracker(client);
                var events = new List<(EventType Type, IAnalyticsEvent Event)>();
                GetAnalyticsTracker(client).OnEventSentDetailed.Where(e => e.Type is EventType.InApp).Subscribe(events.Add);

                // Act
                TrackInApp(client, campaignsTracker, Product1, Product1Name); // First in-app: inAppStatus = NeverPurchased
                TrackInApp(client, campaignsTracker, Product2, Product2Name); // Next in-app: inAppStatus = Purchased

                // Assert
                events.Should().HaveCount(2);
                ((ProductPurchaseEvent)events[0].Event).InAppStatus.Should().Be(InAppStatus.NeverPurchased);
                ((ProductPurchaseEvent)events[1].Event).InAppStatus.Should().Be(InAppStatus.Purchased);
            }

            [Test]
            [UnitTestTarget(typeof(MagifyClient), nameof(MagifyClient.TrackInAppFor))]
            [UnitTestTarget(typeof(MagifyClient), nameof(MagifyClient.TrackIncomeTransaction))]
            public void WhenTrackFirstInAppTransaction_ThenUserIsFree_WhenTrackNextInAppTransaction_ThenUserIsPaying()
            {
                // Arrange
                using var noFailOnError = new TemporaryIgnoreFailingLogMessagesScope();

                using var client = CreateDefaultAndInit();
                var campaignsTracker = GetCampaignsTracker(client);
                var events = new List<(EventType Type, IAnalyticsEvent Event)>();
                GetAnalyticsTracker(client).OnEventSentDetailed.Where(e => e.Type is EventType.Transaction).Subscribe(events.Add);

                // Act
                TrackTransaction(client, TransactionSource); // First transaction: inAppStatus = NeverPurchased
                TrackInApp(client, campaignsTracker, Product1, Product1Name); // First in-app: inAppStatus = NeverPurchased

                TrackTransaction(client, TransactionSource); // Next transaction: inAppStatus = Purchased
                TrackInApp(client, campaignsTracker, Product2, Product2Name); // Next in-app: inAppStatus = Purchased

                // Assert
                events.Should().HaveCount(2);
                ((TransactionEvent)events[0].Event).InAppStatus.Should().Be(InAppStatus.NeverPurchased);
                ((TransactionEvent)events[1].Event).InAppStatus.Should().Be(InAppStatus.Purchased);
            }

            private static void TrackInApp(MagifyClient client, CampaignsTracker campaignsTracker, string productId, string productName, string campaign = Campaign)
            {
                CampaignTrackingUtils.CreateAndTrackClick<InAppCampaign>(campaign, productId, productName, campaignsTracker);
                client.TrackInAppFor(productId, "1", "USD", string.Empty, string.Empty, string.Empty, string.Empty, needValidation: true, PurchaseStore.Native);
            }

            private static void TrackTransaction(MagifyClient client, string source, string bonusName = Bonus1, string productId = Product1)
            {
                var bonuses = new List<BonusInfo>
                {
                    new()
                    {
                        BonusName = Bonus1,
                        Quantity = 1,
                        FinalBalance = 1,
                        Group = "group",
                    },
                };
                var product = new ProductInfo(productId, "1", "USD", PurchaseStore.Native);
                client.TrackIncomeTransaction(source, bonuses, product);
            }
        }
    }
}