using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Rx;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class MagifyClientTests
    {
        public class Context : MagifyClientTestsPart
        {
            [Test]
            [TestCase(ClientKind.Minimal)]
            [TestCase(ClientKind.Full)]
            public async Task WhenSdkFirstlyInited_ThenNoSavedAndDefaultContext_ButThereIsDownloadedContext(ClientKind kind)
            {
                // Arrange
                var configLoaded = false;
                var network = new NetworkMoq((_, message, _) => NetworkMoq.OkResult(message));
                var appliedContexts = new List<ContextKind>();
                using var magifyClient = CreateDefault(kind: kind, networkClient: network);
                var contextApplicator = GetContextApplicator(magifyClient);
                using var _ = contextApplicator.ContextAppliedOnMainThread.Subscribe(appliedContexts.Add);
                magifyClient.OnConfigLoaded += () => configLoaded = true;

                // Act
                var cancellationToken = GetCancellationToken();
                try
                {
                    magifyClient.InitializeSdk();
                    magifyClient.Environment.Value = Environment.Staging;
                    magifyClient.Setup();
                    await TaskHelper.WaitUntil(() => configLoaded, cancellationToken);
                    await UniTask.SwitchToMainThread();
                    await Task.Yield();
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                }

                // Assert
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
                contextApplicator.SavedContextApplied.Should()!.BeTrue();
                contextApplicator.DefaultContextApplied.Should()!.BeTrue();
                contextApplicator.DownloadedContextApplied.Should()!.BeTrue();
                appliedContexts.Should()!.HaveCount(1);
                appliedContexts.Should()!.Contain(ContextKind.Downloaded);
            }

            [Test]
            [TestCase(ClientKind.Minimal)]
            [TestCase(ClientKind.Full)]
            public async Task WhenSdkSecondaryInited_ThenNoDefaultContext_ButThereIsSavedAndDownloadedContext(ClientKind kind)
            {
                // Arrange with first init
                var configLoaded = false;
                var network = new NetworkMoq((_, message, _) => NetworkMoq.OkResult(message));
                var cancellationToken = GetCancellationToken();
                using (var firstMagifyClient = CreateDefault(kind: kind, networkClient: network))
                {
                    firstMagifyClient.OnConfigLoaded += () => configLoaded = true;
                    firstMagifyClient.InitializeSdk();
                    firstMagifyClient.Environment.Value = Environment.Staging;
                    firstMagifyClient.Setup();
                    await TaskHelper.WaitUntil(() => configLoaded, cancellationToken);
                    await UniTask.SwitchToMainThread();
                }

                // Arrange with second init
                configLoaded = false;
                var appliedContexts = new List<ContextKind>();
                using var magifyClient = CreateDefault(networkClient: network);
                var contextApplicator = GetContextApplicator(magifyClient);
                using var _ = contextApplicator.ContextAppliedOnMainThread.Subscribe(appliedContexts.Add);
                magifyClient.OnConfigLoaded += () => configLoaded = true;

                // Act
                try
                {
                    magifyClient.InitializeSdk();
                    magifyClient.Environment.Value = Environment.Staging;
                    magifyClient.Setup();
                    await TaskHelper.WaitUntil(() => configLoaded, cancellationToken);
                    await UniTask.DelayFrame(30, cancellationToken: cancellationToken);
                    await UniTask.SwitchToMainThread();
                    await Task.Yield();
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                }

                // Assert
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
                contextApplicator.SavedContextApplied.Should()!.BeTrue();
                contextApplicator.DefaultContextApplied.Should()!.BeTrue();
                contextApplicator.DownloadedContextApplied.Should()!.BeTrue();
                appliedContexts.Should()!.HaveCount(2);
                appliedContexts.Should()!.HaveElementAt(0, ContextKind.Saved);
                appliedContexts.Should()!.HaveElementAt(1, ContextKind.Downloaded);
            }
        }
    }
}