using System.IO;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    public class FileStorageTests
    {
        private readonly FileStorage _storage = new(Path.Combine(EditorModeTestsEnvironment.RootStoragePath, "test"));

        [SetUp, TearDown]
        public void ClearBeforeAndAfterTests()
        {
            if (Directory.Exists(_storage.RootFolderPath))
            {
                Directory.Delete(_storage.RootFolderPath, true);
            }
        }

        [Test]
        public void WhenSave_AndFileDontExist_ThenShouldCreateNewFile()
        {
            // Arrange
            const string fileName = "file.txt";
            const string data = "Some text";
            _storage.Save(fileName, data);

            // Act
            var filePath = _storage.GetFilePath(fileName, false);
            var exists = File.Exists(filePath);
            var text = File.ReadAllText(filePath);

            // Assert
            exists.Should().BeTrue();
            text.Should().Be(data);
        }

        [Test]
        public void WhenSave_AndFileExist_ThenShouldDeleteOldFileAndCreateNewOne()
        {
            // Arrange
            const string fileName = "file.txt";
            const string firstText = "Some text";
            const string secondText = "Different text";
            _storage.Save(fileName, firstText);
            _storage.Save(fileName, secondText);

            // Act
            var filePath = _storage.GetFilePath(fileName, false);
            var exists = File.Exists(filePath);
            var text = File.ReadAllText(filePath);

            // Assert
            exists.Should().BeTrue();
            text.Should().Be(secondText);
        }

        [Test]
        public void WhenLoad_AndFileDontExist_ThenShouldReturnEmptyString()
        {
            // Arrange
            const string fileName = "file.txt";

            // Act
            var loadText = _storage.Load(fileName);

            // Assert
            loadText.Should().BeEmpty();
        }

        [Test]
        public void WhenLoad_AndFileExist_ThenShouldReturnContentOfFile()
        {
            // Arrange
            const string fileName = "file.txt";
            const string text = "Some text";
            _storage.Save(fileName, text);

            // Act
            var loadText = _storage.Load(fileName);

            // Assert
            loadText.Should().Be(text);
        }
    }
}