﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using FluentAssertions;
using NUnit.Framework;
using static Magify.CounterScopeExt;

namespace Magify.Tests
{
    internal class CountersTests
    {
        private readonly CounterKey _key1 = CounterKey.GetKey(source: "testEvent1");
        private readonly CounterKey _key2 = CounterKey.GetKey(source: "testEvent2");

        [SetUp]
        [TearDown]
        public void CleanStorageBetweenTests()
        {
            EditorModeTestsEnvironment.Clear();
        }

        [TestCase(CounterType.Bonuses, "bonuses")]
        [TestCase(CounterType.Clicks, "clicks")]
        [TestCase(CounterType.Events, "events")]
        [TestCase(CounterType.Impressions, "impressions")]
        [TestCase(CounterType.Rewards, "rewards")]
        [TestCase(CounterType.Nested, "nested")]
        [TestCase(CounterType.LtoStart, "lto_start")]
        public void CounterType_GetDescription_ReturnsCorrectDescription(CounterType scope, string expectedName)
        {
            // Act
            var name = scope.ToEnumString();

            // Assert
            name.Should().Be(expectedName);
        }

        [TestCase(CounterScope.Global, "global")]
        [TestCase(CounterScope.Version, "version")]
        [TestCase(CounterScope.Session, "session")]
        [TestCase(CounterScope.Daily, "daily")]
        [TestCase(CounterScope.Activation, "activation")]
        [TestCase(CounterScope.Period, "period")]
        public void CounterScope_GetName_ReturnsCorrectName(CounterScope scope, string expectedName)
        {
            // Act
            var name = scope.GetName();

            // Assert
            name.Should().Be(expectedName);
        }

        [Test]
        public void CounterScope_GetType_IsUnsignedShort()
        {
            // Arrange
            var counterType = Enum.GetUnderlyingType(typeof(CounterScope));
            var ushortType = typeof(ushort);

            // Assert
            counterType.Should().Be(ushortType);
        }

        [Test]
        public void CounterType_GetType_IsUnsignedShort()
        {
            // Arrange
            var counterType = Enum.GetUnderlyingType(typeof(CounterType));
            var ushortType = typeof(ushort);

            // Assert
            counterType.Should().Be(ushortType);
        }

        [Test]
        public void CounterScope_GetValues_HaveValidNumbering()
        {
            ValidateEnumNumbering2<CounterScope>();
        }

        [Test]
        public void CounterType_GetValues_HaveValidNumbering()
        {
            ValidateEnumNumbering2<CounterType>();
        }

        [Test]
        public void AllCountersTypes_GetDefinitionData_HaveValidNumberingAndNaming()
        {
            // Arrange
            var data = new Dictionary<string, ushort>();
            foreach (var scope in Scopes)
            {
                data[scope.ToString()] = Convert.ToUInt16(scope);
            }

            // Assert
            var countersType = typeof(Counters);
            foreach (var enumType in countersType
                         .GetProperties(BindingFlags.Instance | BindingFlags.Public)
                         .Select(c => c.PropertyType)
                         .Where(c => c.IsGenericType)
                         .Where(c => c.GetGenericTypeDefinition() == typeof(Counter<>))
                         .Select(c => c.GetGenericArguments()[0]))
            {
                foreach (var scope in Enum.GetValues(enumType))
                {
                    var name = scope.ToString();
                    var value = Convert.ToUInt16(scope);

                    data.TryGetValue(name, out var baseValue).Should().Be(true, $"{nameof(CounterScope)} should contains {name}");
                    value.Should().Be(baseValue, $"Numeric representation of {name} in {enumType.Name} should be same as in {nameof(CounterScope)}");
                    Enum.GetUnderlyingType(enumType).Should().Be(typeof(ushort), "All counters scopes should have same underlying type");
                }
            }
        }

        [Test]
        public void CountersContainer_AfterConstructor_HasAllCounters()
        {
            // Arrange
            using var _ = CreateCounters(out var counters);

            // Assert
            foreach (var counterType in EnumExtensions.GetValues<CounterType>())
            {
                counters[counterType].Should().NotBeNull($"{nameof(Counters)} should contains {counterType} after creation");
            }
        }

        [Test]
        public void CountersContainer_AfterConstructor_AllCounterInitialized()
        {
            // Arrange
            using var _ = CreateCounters(out var counters);

            // Assert
            var countersType = typeof(Counters);
            foreach (var property in countersType
                         .GetProperties(BindingFlags.Instance | BindingFlags.Public)
                         .Where(c => c.PropertyType.IsGenericType)
                         .Where(c => c.PropertyType.GetGenericTypeDefinition() == typeof(Counter<>)))
            {
                property.GetValue(counters).Should().NotBeNull($"{nameof(Counters)}.{property.Name} should be initialized after creation");
            }
        }

        [Test]
        public void CounterScope_GetAndSet_AreTheSame()
        {
            // Arrange
            using var _ = CreateCounter<CounterScope>(out var counter);

            // Act
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1] = i;
            }

            // Assert
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1].Should().Be(i);
            }
        }

        [Test]
        public void CounterScope_MethodIncrement_IncrementsScopeByOne()
        {
            // Arrange
            using var _ = CreateCounter<CounterScope>(out var counter);
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1] = i;
            }

            // Act
            foreach (var scope in Scopes)
            {
                counter.Increment(scope, _key1);
            }

            // Assert
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1].Should().Be(i + 1);
            }
        }

        [Test]
        public void CounterScope_MethodIncrementAll_IncrementsAllScopesByOne()
        {
            // Arrange
            using var _ = CreateCounter<CounterScope>(out var counter);
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1] = i;
            }

            // Act
            counter.IncrementAll(_key1);

            // Assert
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1].Should().Be(i + 1);
            }
        }

        [Test]
        public void CounterScope_MethodDecrement_DecrementsScopeByOne()
        {
            // Arrange
            using var _ = CreateCounter<CounterScope>(out var counter);
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1] = i + 1;
            }

            // Act
            foreach (var scope in Scopes)
            {
                counter.Decrement(scope, _key1);
            }

            // Assert
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1].Should().Be(i);
            }
        }

        [Test]
        public void CounterScope_MethodDecrementForZeroValue_DoesNotChangeScope()
        {
            // Arrange
            using var _ = CreateCounter<CounterScope>(out var counter);
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1] = 0;
            }

            // Act
            foreach (var scope in Scopes)
            {
                counter.Decrement(scope, _key1);
            }

            // Assert
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1].Should().Be(0);
            }
        }

        [Test]
        public void CounterScope_MethodDecrementAll_DecrementsAllScopesByOne()
        {
            // Arrange
            using var _ = CreateCounter<CounterScope>(out var counter);
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1] = i + 1;
            }

            // Act
            counter.DecrementAll(_key1);

            // Assert
            for (var i = 0; i < Scopes.Length; i++)
            {
                counter[Scopes[i], _key1].Should().Be(i);
            }
        }

        [Test]
        public void CounterScope_MethodResetWithScopeAndEntity_ResetsOnlySpecificEntityInSpecificScope()
        {
            // Arrange
            using var _ = CreateCounter<CounterScope>(out var counter);
            counter[CounterScope.Global, _key1] = 10;
            counter[CounterScope.Version, _key1] = 15;
            counter[CounterScope.Session, _key1] = 18;
            counter[CounterScope.Daily, _key1] = 23;
            counter[CounterScope.Session, _key2] = 21;

            // Act
            counter.Reset(CounterScope.Session, _key1);

            // Assert
            counter[CounterScope.Global, _key1].Should().Be(10);
            counter[CounterScope.Version, _key1].Should().Be(15);
            counter[CounterScope.Session, _key1].Should().Be(0);
            counter[CounterScope.Daily, _key1].Should().Be(23);
            counter[CounterScope.Session, _key2].Should().Be(21);
        }

        [Test]
        public void CounterScope_MethodResetWithScope_ResetsAllEntitiesInSpecificScope()
        {
            // Arrange
            using var _ = CreateCounter<CounterScope>(out var counter);
            counter[CounterScope.Global, _key1] = 10;
            counter[CounterScope.Global, _key2] = 20;
            counter[CounterScope.Version, _key1] = 30;
            counter[CounterScope.Version, _key2] = 40;

            // Act
            counter.ResetAll(CounterScope.Global);

            // Assert
            counter[CounterScope.Global, _key1].Should().Be(0);
            counter[CounterScope.Global, _key2].Should().Be(0);
            counter[CounterScope.Version, _key1].Should().Be(30);
            counter[CounterScope.Version, _key2].Should().Be(40);
        }

        private void ValidateEnumNumbering2<TEnum>()
            where TEnum : Enum
        {
            var names = Enum.GetNames(typeof(TEnum));
            var values = Enum.GetValues(typeof(TEnum)).Cast<TEnum>().ToList();
            for (var i = 0; i < values.Count; i++)
            {
                var member = Convert.ToInt32(values[i]);
                member.Should().Be(i, $"[{names[i]} should be with number {i} but it is {member}]");
            }
        }

        private static SubsystemsCollection CreateCounters(out Counters counters)
        {
            var subsystems = new SubsystemsCollection();
            var storage = new CountersStorage(EditorModeTestsEnvironment.CountersFolderPath).AddTo(subsystems);
            counters = new Counters(storage).AddTo(subsystems);
            return subsystems;
        }

        private static SubsystemsCollection CreateCounter<T>(out Counter<T> counter)
            where T : unmanaged, Enum
        {
            var subsystems = new SubsystemsCollection();
            var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.CountersFolderPath)
                .AddEntry(BinaryTypeString.Shared)
                .AddEntry(BinaryTypeInt32.Shared)
                .SupportDictionariesOf<string, int>()
                .Build()
                .AddTo(subsystems);
            counter = new Counter<T>(storage).AddTo(subsystems);
            return subsystems;
        }
    }
}