﻿using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class ContextListenerTests
    {
        internal class AssignedAbTestsTests
        {
            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContext_ThenAbTestsShouldBeSameAbTestsCount(ContextKind kind)
            {
                //Arrange
                var listener = new AssignedAbTests();
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)listener).UpdateContext(context, kind);

                //Assert
                listener.Values!.Value!.Count.Should()!.Be(context.AssignedAbTests!.Count);
            }

            [Test]
            [TestCase(ContextKind.Default)]
            public void WhenUpdateDefaultKindContext_ThenAbTestsShouldBeEmpty(ContextKind kind)
            {
                //Arrange
                var listener = new AssignedAbTests();
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)listener).UpdateContext(context, kind);

                //Assert
                listener.Values!.Value!.Should()!.BeEmpty();
            }

            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextMultiplyTimes_ThenUpdatedAbTestsShouldBeSameAsContext(ContextKind kind)
            {
                //Arrange
                var listener = new Magify.AssignedAbTests();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();

                //Act
                ((IContextListener)listener).UpdateContext(firstContext, kind);
                var firstUpdateAbTests = listener.Values!.Value!;
                ((IContextListener)listener).UpdateContext(secondContext, kind);
                var secondUpdateAbTests  = listener.Values!.Value!;

                //Assert
                firstContext.Should()!.NotBe(secondContext);
                firstUpdateAbTests.Count.Should()!.Be(firstContext.AssignedAbTests?.Count);
                secondUpdateAbTests.Count.Should()!.Be(secondContext.AssignedAbTests?.Count);
            }

            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public async Task WhenUpdateContextOnThreadPool_ThenUpdateContextShouldBeThreadSafety(ContextKind kind)
            {
                //Arrange
                var listener = new Magify.AssignedAbTests();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();
                IReadOnlyList<AssignedAbTest> firstUpdateAbTests = default;
                IReadOnlyList<AssignedAbTest> secondUpdateAbTests = default;
                var barrier = new Barrier(2);

                //Act
                var firstTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)listener).UpdateContext(firstContext, kind);
                    firstUpdateAbTests = listener.Values!.Value!;
                });

                var secondTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)listener).UpdateContext(secondContext, kind);
                    secondUpdateAbTests = listener.Values!.Value!;
                });

                await UniTask.WhenAll(firstTask, secondTask);

                //Assert
                firstUpdateAbTests.Count.Should()!.Be(firstContext.AssignedAbTests?.Count);
                firstUpdateAbTests.Select(c => c?.Name).ToArray()
                    .ShouldBeSameInAnyOrder(firstContext.AssignedAbTests?.Select(c => c?.Name).ToArray());

                secondUpdateAbTests.Count.Should()!.Be(secondContext.AssignedAbTests?.Count);
                secondUpdateAbTests.Select(c => c?.Name).ToArray()
                    .ShouldBeSameInAnyOrder(secondContext.AssignedAbTests?.Select(c => c?.Name).ToArray());
            }
        }
    }
}