﻿using System;
using System.IO;
using System.Threading;
using System.Threading.Tasks;
using FluentAssertions;
using JetBrains.Annotations;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    internal class AppStateBuilderTests
    {
        [NotNull]
        internal static object[] AppStateTestCases()
        {
            return new object[]
            {
                "client data, hope that everything is ok",
                "a",
                "just another string",
                string.Empty,
            };
        }

        [Test]
        [TestCaseSource(nameof(AppStateTestCases))]
        public async Task Encode_Decode([NotNull] string clientState)
        {
            // Arrange
            using var _ = CreateAppStateBuilder(out var appStateBuilder);
            File.WriteAllText(EditorModeTestsEnvironment.ClientStatePath, clientState);

            // Act
            var encodedAppState = await appStateBuilder.Build(CancellationToken.None, RuntimePlatform.Android);
            var decodedAppState = appStateBuilder.Parse(encodedAppState, RuntimePlatform.Android);

            // Assert
            decodedAppState.ClientState.Should().Be(clientState);
        }

        [Test]
        [TestCaseSource(nameof(AppStateTestCases))]
        public async Task EncodeInOneMode_DecodeWithAnotherSet([NotNull] string clientState)
        {
            // Arrange
            using var _ = CreateAppStateBuilder(out var appStateBuilder);
            File.WriteAllText(EditorModeTestsEnvironment.ClientStatePath, clientState);

            // Act
            appStateBuilder.EncodingMode = EncodingMode.ToJsonToZipToBase64;
            var encodedAppState = await appStateBuilder.Build(CancellationToken.None, RuntimePlatform.Android);
            appStateBuilder.EncodingMode = EncodingMode.ToJsonToBase64;
            var decodedAppState = appStateBuilder.Parse(encodedAppState, RuntimePlatform.Android);

            // Assert
            decodedAppState.ClientState.Should().Be(clientState);
        }

        [NotNull]
        private static IDisposable CreateAppStateBuilder([NotNull] out AppStateBuilder appStateBuilder)
        {
            var subsystems = new SubsystemsCollection();
            var platform = new PlatformDefault().AddTo(subsystems);
            var generalPrefs = CreateGeneralPrefs(platform).AddTo(subsystems);
            var counters = CreateCountersStorage().AddTo(subsystems);
            appStateBuilder = new AppStateBuilder(
                generalPrefs,
                counters,
                new ClientStateConfig(new AppStateFromTestFileProvider(EditorModeTestsEnvironment.ClientStatePath)),
                EditorModeTestsEnvironment.RootStoragePath);
            return subsystems;
        }

        [NotNull]
        private static GeneralPrefs CreateGeneralPrefs([NotNull] PlatformAPI platform)
        {
            return GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, new AppVersionProvider());
        }

        [NotNull]
        private static CountersStorage CreateCountersStorage()
        {
            return new CountersStorage(EditorModeTestsEnvironment.CountersFolderPath);
        }
    }
}