using System.IO;
using System.Text;
using FluentAssertions;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    public class BinaryTypeHandlerTests
    {
        private readonly byte[] _buffer = new byte[4096];

        private static Vector2[] Vector2Inputs => new[]
        {
            Vector2.left, Vector2.right, Vector2.down, Vector2.up, Vector2.zero, Vector2.negativeInfinity, Vector2.one,
        };

        private static Vector3[] Vector3Inputs => new[]
        {
            Vector3.left, Vector3.right, Vector3.down, Vector3.up, Vector3.zero, Vector3.negativeInfinity, Vector3.one,
        };

        private static byte[][] ByteArrayInputs => new[]
        {
            new byte[] { byte.MinValue, 1, 2, 3, byte.MaxValue },
            null
        };

        [TestCase(false)]
        [TestCase(true)]
        public void WhenBinaryTypeBooleanUsed_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(bool value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeBoolean, bool>(value);
        }

        [TestCase('\0')]
        [TestCase('\n')]
        [TestCase('g')] // latin
        [TestCase('г')] // cyrillic
        [TestCase('里')] // chinese simplified
        [TestCase('爾')] // chinese traditional
        [TestCase('ة')] // arabic
        public void WhenBinaryTypeCharUsed_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(char value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeChar, char>(value);
        }

        [TestCase(null)] // null
        [TestCase("")] // empty
        [TestCase("Hello world!")] // latin
        [TestCase("Прывітанне сусвет!")] // cyrillic
        [TestCase("你好世界")] // chinese
        [TestCase("مرحبا بالعالم!")] // arabic
        public void WhenBinaryTypeStringUsed_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(string value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeString, string>(value);
        }

        [TestCase(byte.MinValue)]
        [TestCase(byte.MaxValue)]
        public void WhenBinaryTypeByteUsed_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(byte value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeByte, byte>(value);
        }

        [TestCase(sbyte.MinValue)]
        [TestCase(0)]
        [TestCase(sbyte.MaxValue)]
        public void WhenBinaryTypeSByteUsed_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(sbyte value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeSByte, sbyte>(value);
        }

        [TestCase(short.MinValue)]
        [TestCase(0)]
        [TestCase(short.MaxValue)]
        public void WhenBinaryTypeInt16Used_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(short value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeInt16, short>(value);
        }

        [TestCase(ushort.MinValue)]
        [TestCase(ushort.MaxValue)]
        public void WhenBinaryTypeUInt16Used_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(ushort value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeUInt16, ushort>(value);
        }

        [TestCase(int.MinValue)]
        [TestCase(0)]
        [TestCase(int.MaxValue)]
        public void WhenBinaryTypeInt32Used_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(int value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeInt32, int>(value);
        }

        [TestCase(uint.MinValue)]
        [TestCase(uint.MaxValue)]
        public void WhenBinaryTypeUInt32Used_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(uint value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeUInt32, uint>(value);
        }

        [TestCase(long.MinValue)]
        [TestCase(0)]
        [TestCase(long.MaxValue)]
        public void WhenBinaryTypeInt64Used_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(long value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeInt64, long>(value);
        }

        [TestCase(ulong.MinValue)]
        [TestCase(ulong.MaxValue)]
        public void WhenBinaryTypeUInt64Used_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(ulong value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeUInt64, ulong>(value);
        }

        [TestCase(float.MinValue)]
        [TestCase(0f)]
        [TestCase(float.MaxValue)]
        public void WhenBinaryTypeSingleUsed_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(float value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeSingle, float>(value);
        }

        [TestCase(double.MinValue)]
        [TestCase(0.0)]
        [TestCase(double.MaxValue)]
        public void WhenBinaryTypeDoubleUsed_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(double value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeDouble, double>(value);
        }

        [TestCase(int.MinValue)]
        [TestCase(0.0)]
        [TestCase(int.MaxValue)]
        public void WhenBinaryTypeDecimalUsed_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(decimal value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeDecimal, decimal>(value);
        }

        [TestCaseSource(nameof(Vector2Inputs))]
        public void WhenBinaryTypeVector2Used_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(Vector2 value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeVector2, Vector2>(value);
        }

        [TestCaseSource(nameof(Vector3Inputs))]
        public void WhenBinaryTypeVector3Used_AndWrittenToStream_ThenSizeOfBufferAndReadValueAreCorrect(Vector3 value)
        {
            DoSizeCheckForBinaryTypeHandler<BinaryTypeVector3, Vector3>(value);
        }

        private void DoSizeCheckForBinaryTypeHandler<T, TType>(TType value)
            where T : BinaryTypeHandler<TType>, new()
        {
            // Arrange
            var binaryType = new T();
            var size = binaryType.SizeOf(value);
            using var writeStream = new MemoryStream(_buffer);
            using var readStream = new MemoryStream(_buffer);
            using var writer = new BinaryWriter(writeStream, Encoding.Unicode);
            using var reader = new BinaryReader(readStream, Encoding.Unicode);

            // Act
            binaryType.WriteTo(writer, value);

            // Assert
            writeStream.Position.Should().Be(size);
            readStream.Position.Should().Be(0);
            binaryType.ReadFrom(reader).Should().Be(value);
        }
    }
}