﻿using System;
using System.Reflection;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    public class BinaryStorageNullableTests
    {
        [Test]
        [TestCase(typeof(bool), null, false)]
        [TestCase(typeof(byte), null, byte.MinValue)]
        [TestCase(typeof(sbyte), null, sbyte.MinValue)]
        [TestCase(typeof(int), null, int.MinValue)]
        [TestCase(typeof(uint), null, uint.MinValue)]
        [TestCase(typeof(long), null, long.MinValue)]
        [TestCase(typeof(ulong), null, ulong.MinValue)]
        [TestCase(typeof(float), null, float.MinValue)]
        [TestCase(typeof(double), null, double.MinValue)]
        [TestCase(typeof(char), null, char.MinValue)]
        public void Test(Type genericType, object value1, object value2)
        {
            method(nameof(SetGetFor)).Invoke(null, new[] { value1 });
            method(nameof(SetGetFor)).Invoke(null, new[] { value2 });
            method(nameof(SaveLoadFor)).Invoke(null, new[] { value1 });
            method(nameof(SaveLoadFor)).Invoke(null, new[] { value2 });
            method(nameof(SetGetTwiceFor)).Invoke(null, new[] { value1, value2 });
            method(nameof(SetGetTwiceFor)).Invoke(null, new[] { value2, value1 });
            method(nameof(SaveLoadTwiceFor)).Invoke(null, new[] { value1, value2 });
            method(nameof(SaveLoadTwiceFor)).Invoke(null, new[] { value2, value1 });

            MethodInfo method(string name)
            {
                return typeof(BinaryStorageNullableTests)
                    .GetMethod(name, BindingFlags.NonPublic | BindingFlags.Static)
                    !.MakeGenericMethod(genericType);
            }
        }

        private static void SetGetFor<T>(T? value)
            where T: struct
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .SupportNullable<T>()
                .Build();

            // Act
            storage.Set<T?>("key", value);

            // Assert
            storage.Get<T?>("key").Should().Be(value);

            // Cleanup
            storage.ResetAll();
        }

        private static void SaveLoadFor<T>(T? value)
            where T: struct
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .SupportNullable<T>()
                .Build();

            // Act
            storage.Set<T?>("key", value);
            storage.SaveDataOnDisk();

            // Assert
            storage.ResetAll();
            storage.LoadDataFromDisk();
            storage.Get<T?>("key").Should().Be(value);

            // Cleanup
            storage.ResetAll();
        }

        private static void SetGetTwiceFor<T>(T? value1, T? value2)
            where T: struct
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .SupportNullable<T>()
                .Build();

            // Act
            storage.Set<T?>("key", value1);
            var readValue1 = storage.Get<T?>("key");
            storage.Set<T?>("key", value2);
            var readValue2 = storage.Get<T?>("key");

            // Assert
            readValue1.Should().Be(value1);
            readValue2.Should().Be(value2);

            // Cleanup
            storage.ResetAll();
        }

        private static void SaveLoadTwiceFor<T>(T? value1, T? value2)
            where T: struct
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .SupportNullable<T>()
                .Build();

            // Act
            storage.Set<T?>("key", value1);
            storage.SaveDataOnDisk();
            storage.ResetAll();
            storage.LoadDataFromDisk();
            var readValue1 = storage.Get<T?>("key");

            storage.Set<T?>("key", value2);
            storage.SaveDataOnDisk();
            storage.ResetAll();
            storage.LoadDataFromDisk();
            var readValue2 = storage.Get<T?>("key");

            // Assert
            readValue1.Should().Be(value1);
            readValue2.Should().Be(value2);

            // Cleanup
            storage.ResetAll();
        }
    }
}