﻿using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    public class BinaryCollectionTests
    {
        private readonly byte[] _buffer = new byte[4096];

        private static string[] StringsInputs => new[]
        {
            "", "Hello world!", "Прывітанне сусвет!", "你好世界", "مرحبا بالعالم!"
        };

        [SetUp]
        [TearDown]
        public void CleanStorageBetweenTests()
        {
            EditorModeTestsEnvironment.Clear();
        }

        #region ReactiveList

        [Test]
        public void WhenReactiveListSerialized_AndGeneralListSerialized_ThenJsonsAreEquals()
        {
            // Arrange
            var collection1 = new List<string>();
            var collection2 = new ReactiveList<string>();
            foreach (var input in StringsInputs)
            {
                collection1.Add(input);
                collection2.Add(input);
            }

            // Act
            var json1 = JsonFacade.SerializeObject(collection1);
            var json2 = JsonFacade.SerializeObject(collection2);

            // Assert
            json1.Should().Be(json2);
        }

        [Test]
        public void WhenReactiveListUsed_AndWrittenToStream_ThenReadValuesFromStreamAreCorrect()
        {
            var values = StringsInputs;

            // Arrange
            var binaryType = BinaryTypeString.Shared;
            var section = new CollectionSection<string, ReactiveList<string>>(binaryType);
            var set = section.Get("key");
            foreach (var value in values)
            {
                set.Add(value);
            }

            using var writeStream = new MemoryStream(_buffer);
            using var readStream = new MemoryStream(_buffer);
            using var writer = new BinaryWriter(writeStream, Encoding.Unicode);
            using var reader = new BinaryReader(readStream, Encoding.Unicode);

            // Act
            section.WriteTo(writer);
            section.ReadFrom(reader);

            // Assert
            var deserializedSection = section.Get("key");
            deserializedSection.Count.Should().Be(set.Count);
            deserializedSection.Should().BeEquivalentTo(set);
        }

        [Test]
        public void WhenReactiveListChanged_AndReloaded_ThenAllDataAreValid()
        {
            // Arrange
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeString.Shared)
                       .SupportListsOf<string>()
                       .Build())
            {
                var strings = storage.GetList<string>("key");
                using (storage.MultipleChangeScope())
                {
                    foreach (var str in StringsInputs)
                    {
                        strings.Add(str);
                    }
                }
            }

            // Act
            // Assert
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeString.Shared)
                       .SupportListsOf<string>()
                       .Build())
            {
                var strings = storage.GetList<string>("key");
                strings.Should().BeEquivalentTo(new List<string>(StringsInputs));
            }
        }

        #endregion

        #region ReactiveSet

        [Test]
        public void WhenReactiveSetSerialized_AndGeneralSetSerialized_ThenJsonsAreEquals()
        {
            // Arrange
            var collection1 = new HashSet<string>();
            var collection2 = new ReactiveHashSet<string>();
            foreach (var input in StringsInputs)
            {
                collection1.Add(input);
                collection2.Add(input);
            }

            // Act
            var json1 = JsonFacade.SerializeObject(collection1);
            var json2 = JsonFacade.SerializeObject(collection2);

            // Assert
            json1.Should().Be(json2);
        }

        [Test]
        public void WhenReactiveSetUsed_AndWrittenToStream_ThenReadValuesFromStreamAreCorrect()
        {
            var values = StringsInputs;

            // Arrange
            var binaryType = BinaryTypeString.Shared;
            var section = new CollectionSection<string, ReactiveHashSet<string>>(binaryType);
            var set = section.Get("key");
            foreach (var value in values)
            {
                set.Add(value);
            }

            using var writeStream = new MemoryStream(_buffer);
            using var readStream = new MemoryStream(_buffer);
            using var writer = new BinaryWriter(writeStream, Encoding.Unicode);
            using var reader = new BinaryReader(readStream, Encoding.Unicode);

            // Act
            section.WriteTo(writer);
            section.ReadFrom(reader);

            // Assert
            var deserializedSection = section.Get("key");
            deserializedSection.Count.Should().Be(set.Count);
            deserializedSection.Should().BeEquivalentTo(set);
        }

        [Test]
        public void WhenReactiveSetChanged_AndReloaded_ThenAllDataAreValid()
        {
            // Arrange
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeString.Shared)
                       .SupportSetsOf<string>()
                       .Build())
            {
                var strings = storage.GetSet<string>("key");
                using (storage.MultipleChangeScope())
                {
                    foreach (var str in StringsInputs)
                    {
                        strings.Add(str);
                    }
                }
            }

            // Act
            // Assert
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeString.Shared)
                       .SupportSetsOf<string>()
                       .Build())
            {
                var strings = storage.GetSet<string>("key");
                strings.Should().BeEquivalentTo(new HashSet<string>(StringsInputs));
            }
        }

        #endregion

        #region ReactiveDictionary

        [Test]
        public void WhenReactiveDictionarySerialized_AndGeneralDictionarySerialized_ThenJsonsAreEquals()
        {
            // Arrange
            var collection1 = new Dictionary<string, string>();
            var collection2 = new ReactiveDictionary<string, string>();
            foreach (var input in StringsInputs)
            {
                collection1[input] = $"value_{input}";
                collection2[input] = $"value_{input}";
            }

            // Act
            var json1 = JsonFacade.SerializeObject(collection1);
            var json2 = JsonFacade.SerializeObject(collection2);

            // Assert
            json1.Should().Be(json2);
        }

        [Test]
        public void WhenReactiveDictionaryUsed_AndWrittenToStream_ThenReadValuesFromStreamAreCorrect()
        {
            var keys = StringsInputs;

            // Arrange
            var binaryType = new BinaryTypesPair<string, string>(BinaryTypeString.Shared, BinaryTypeString.Shared);
            var section = new CollectionSection<KeyValuePair<string, string>, ReactiveDictionary<string, string>>(binaryType);
            var set = section.Get("key");
            foreach (var key in keys)
            {
                set.Add(key, "value");
            }

            using var writeStream = new MemoryStream(_buffer);
            using var readStream = new MemoryStream(_buffer);
            using var writer = new BinaryWriter(writeStream, Encoding.Unicode);
            using var reader = new BinaryReader(readStream, Encoding.Unicode);

            // Act
            section.WriteTo(writer);
            section.ReadFrom(reader);

            // Assert
            var deserializedSection = section.Get("key");
            deserializedSection.Count.Should().Be(set.Count);
            deserializedSection.Should().BeEquivalentTo(set);
        }

        [Test]
        public void WhenReactiveDictionaryChanged_AndReloaded_ThenAllDataAreValid()
        {
            // Arrange
            var keys = StringsInputs;
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeString.Shared)
                       .SupportDictionariesOf<string, string>()
                       .Build())
            {
                var strings = storage.GetDictionary<string, string>("key");
                using (storage.MultipleChangeScope())
                {
                    foreach (var key in keys)
                    {
                        strings.Add(key, "value");
                    }
                }
            }

            // Act
            // Assert
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeString.Shared)
                       .SupportDictionariesOf<string, string>()
                       .Build())
            {
                var strings = storage.GetDictionary<string, string>("key");
                strings.Should().BeEquivalentTo(new Dictionary<string, string>(StringsInputs.ToDictionary(c => c, _ => "value")));
            }
        }

        #endregion
    }
}