using System;
using System.Threading;
using Cysharp.Threading.Tasks;

namespace Magify
{
    internal static class UniTaskExtensions
    {
        public static bool IsCompleted(this UniTaskCompletionSource task)
        {
            return task.UnsafeGetStatus().IsCompleted();
        }

        public static bool IsCompleted<T>(this UniTaskCompletionSource<T> task)
        {
            return task.UnsafeGetStatus().IsCompleted();
        }

        public static bool InProgress(this UniTaskCompletionSource task)
        {
            return task.UnsafeGetStatus() == UniTaskStatus.Pending;
        }

        public static bool InProgress<T>(this UniTaskCompletionSource<T> task)
        {
            return task.UnsafeGetStatus() == UniTaskStatus.Pending;
        }

        public static UniTask<T> RepeatWhile<T>(this Func<UniTask<T>> task, Func<T, bool> endCondition, CancellationToken cancellationToken)
        {
            cancellationToken.ThrowIfCancellationRequested();
            var promise = new UniTaskCompletionSource<T>();
            cancellationToken.Register(() => promise.TrySetCanceled(cancellationToken));
            routine().Forget();
            return promise.Task;

            async UniTaskVoid routine()
            {
                while (true)
                {
                    var result = await task();
                    if (cancellationToken.IsCancellationRequested)
                    {
                        promise.TrySetCanceled(cancellationToken);
                        return;
                    }
                    if (endCondition(result))
                    {
                        promise.TrySetResult(result);
                        break;
                    }
                }
            }
        }

        public static UniTask RepeatWhile(this Func<UniTask> task, Func<bool> endCondition, CancellationToken cancellationToken, ExponentialBackoff backoff = null)
        {
            if (cancellationToken.IsCancellationRequested)
            {
                return UniTask.FromCanceled(cancellationToken);
            }
            var promise = new UniTaskCompletionSource();
            cancellationToken.Register(() => promise.TrySetCanceled(cancellationToken));
            routine().Forget();
            return promise.Task;

            async UniTaskVoid routine()
            {
                while (true)
                {
                    await task();
                    if (cancellationToken.IsCancellationRequested)
                    {
                        promise.TrySetCanceled(cancellationToken);
                        return;
                    }
                    if (endCondition())
                    {
                        promise.TrySetResult();
                        break;
                    }
                    if (backoff != null)
                    {
                        await UniTask.Delay(TimeSpan.FromMilliseconds(backoff.NextDelay()), true);
                        if (cancellationToken.IsCancellationRequested)
                        {
                            promise.TrySetCanceled(cancellationToken);
                            return;
                        }
                    }
                }
            }
        }
    }
}