﻿using System.Text.RegularExpressions;

namespace Magify
{
    public static class LanguageTagUtils
    {
        /// <summary>
        /// Checks if language tag is valid. If not, tries to fix it with <see cref="OnlyLanguageToFull"/> method.
        /// </summary>
        public static bool TryFixLanguageTag(string languageTag, out string fixedLanguageTag)
        {
            if (IsValid(languageTag))
            {
                fixedLanguageTag = languageTag;
                return true;
            }
            if (HasOnlyLanguage(languageTag))
            {
                fixedLanguageTag = OnlyLanguageToFull(languageTag);
                return true;
            }
            fixedLanguageTag = languageTag;
            return false;
        }

        /// <summary>
        /// Uses pattern: "^[a-z]+-[A-Z]+[a-z]*$"
        /// </summary>
        /// <example>
        /// "us-EN" is valid <br/>
        /// "us-En" is valid <br/>
        /// "zh-Hans" is valid <br/>
        /// "zh-Hant" is valid <br/>
        /// "us-en" is invalid <br/>
        /// "US-En" is invalid <br/>
        /// "us-" is invalid <br/>
        /// "-EN" is invalid <br/>
        /// "us" is invalid <br/>
        /// "EN" is invalid <br/>
        /// "-" is invalid
        /// "" is invalid
        /// </example>
        public static bool IsValid(string languageTag)
        {
            return new Regex("^[a-z]+-[A-Z]+[a-z]*$").IsMatch(languageTag);
        }

        /// <summary>
        /// Uses pattern: "^[a-z]+$" || "^[a-z]+-$"
        /// </summary>
        /// <returns>
        /// "us" is true <br/>
        /// "us-" is true
        /// </returns>
        public static bool HasOnlyLanguage(string languageTag)
        {
            return new Regex("^[a-z]+$").IsMatch(languageTag)
                || new Regex("^[a-z]+-$").IsMatch(languageTag);
        }

        /// <summary>
        /// Should call this method only if <see cref="HasOnlyLanguage"/> returns true. Otherwise, it might return weird results.
        /// </summary>
        /// <returns>
        /// New string with language tag and generated region: <br/>
        /// "us" -> "us-US" <br/>
        /// "us-" -> "us-US"
        /// </returns>
        public static string OnlyLanguageToFull(string languageTag)
        {
            var splitter = languageTag.IndexOf('-');
            return splitter == -1
                ? $"{languageTag}-{languageTag.ToUpperInvariant()}"
                : $"{languageTag}{languageTag[..splitter].ToUpperInvariant()}";
        }
    }
}